import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { motion, AnimatePresence } from 'framer-motion';
import { BaseCrudService } from '@/integrations';
import { useMember } from '@/integrations';
import { AnimeTitles, MangaTitles, MangaChapters, MangaVolumes, Yorumlar, KullancProfilleri, Haberler, Notifications } from '@/entities';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Image as ImageComponent } from '@/components/ui/image';
import { ServerVideoStatus } from '@/components/ui/server-video-status';
import { 
  Users, 
  BookOpen, 
  Play, 
  MessageSquare, 
  Plus, 
  Edit, 
  Trash2, 
  Eye,
  BarChart3,
  Star,
  LogOut,
  Save,
  X,
  TrendingUp,
  Activity,
  Calendar,
  Clock,
  FileVideo,
  FileText,
  Shield,
  Settings,
  Image,
  AlertCircle,
  CheckCircle,
  Loader2,
  Newspaper,
  Bell,
  Server
} from 'lucide-react';

// Video type detection for admin preview
const getVideoType = (url: string): 'youtube' | 'googledrive' | 'invalid' => {
  if (!url) return 'invalid';
  if (url.includes('youtube.com') || url.includes('youtu.be')) {
    return 'youtube';
  }
  if (url.includes('drive.google.com')) {
    return 'googledrive';
  }
  return 'invalid';
};

// Video platform badge component
const VideoPlatformBadge = ({ videoUrl }: { videoUrl: string }) => {
  const videoType = getVideoType(videoUrl);
  
  switch (videoType) {
    case 'youtube':
      return (
        <Badge className="bg-red-500/20 text-red-400 border-red-500/30">
          <FileVideo className="w-3 h-3 mr-1" />
          YouTube
        </Badge>
      );
    case 'googledrive':
      return (
        <Badge className="bg-blue-500/20 text-blue-400 border-blue-500/30">
          <FileVideo className="w-3 h-3 mr-1" />
          Google Drive
        </Badge>
      );
    default:
      return null;
  }
};
// Enhanced Image Preview Component 
const ImagePreview = ({ 
  src,
  alt, 
  width = 120, 
  height = 160, 
  className = "",
  borderColor = "accentneongreen"
}: {
  src: string;
  alt: string;
  width?: number;
  height?: number;
  className?: string;
  borderColor?: string;
}) => {
  const [imageStatus, setImageStatus] = useState<'loading' | 'loaded' | 'error'>('loading');
  const [imageSrc, setImageSrc] = useState<string>('');

  useEffect(() => {
    if (!src || src.trim() === '') {
      setImageStatus('error');
      return;
    }

    setImageStatus('loading');
    setImageSrc('');

    // Create a new image to test if the URL is valid
    const img = new window.Image();
    
    img.onload = () => {
      setImageSrc(src);
      setImageStatus('loaded');
    };
    
    img.onerror = () => {
      setImageStatus('error');
    };

    // Add a timeout to prevent infinite loading
    const timeout = setTimeout(() => {
      if (imageStatus === 'loading') {
        setImageStatus('error');
      }
    }, 10000); // 10 second timeout

    img.src = src;

    return () => {
      clearTimeout(timeout);
      img.onload = null;
      img.onerror = null;
    };
  }, [src]);

  return (
    <motion.div
      initial={{ opacity: 0, scale: 0.9 }}
      animate={{ opacity: 1, scale: 1 }}
      className={`mt-3 p-3 bg-primary/30 rounded-lg border border-${borderColor}/20 ${className}`}
    >
      <p className="text-xs text-primary-foreground/60 mb-2">Önizleme:</p>
      
      <div 
        className={`relative rounded-lg border border-${borderColor}/30 overflow-hidden bg-primary/20`}
        style={{ width: `${width}px`, height: `${height}px` }}
      >
        {imageStatus === 'loading' && (
          <div className="absolute inset-0 flex items-center justify-center">
            <div className="flex flex-col items-center space-y-2">
              <Loader2 className={`w-6 h-6 text-${borderColor} animate-spin`} />
              <span className="text-xs text-primary-foreground/60">Yükleniyor...</span>
            </div>
          </div>
        )}
        
        {imageStatus === 'loaded' && imageSrc && (
          <>
            <ImageComponent
              src={imageSrc}
              alt={alt}
              width={width}
              height={height}
              className="w-full h-full object-cover"
            />
            <div className="absolute top-2 right-2">
              <div className={`p-1 bg-${borderColor}/20 rounded-full`}>
                <CheckCircle className={`w-3 h-3 text-${borderColor}`} />
              </div>
            </div>
          </>
        )}
        
        {imageStatus === 'error' && (
          <div className="absolute inset-0 flex items-center justify-center">
            <div className="flex flex-col items-center space-y-2 text-center p-2">
              <AlertCircle className="w-6 h-6 text-red-400" />
              <span className="text-xs text-red-400">Görsel yüklenemedi</span>
              <span className="text-xs text-primary-foreground/40">URL'yi kontrol edin</span>
            </div>
          </div>
        )}
      </div>
      
      {imageStatus === 'loaded' && (
        <div className="mt-2 flex items-center space-x-2">
          <CheckCircle className={`w-3 h-3 text-${borderColor}`} />
          <span className={`text-xs text-${borderColor}`}>Görsel başarıyla yüklendi</span>
        </div>
      )}
    </motion.div>
  );
};

export default function AdminPage() {
  const { member, actions } = useMember();
  const [animeList, setAnimeList] = useState<AnimeTitles[]>([]);
  const [mangaList, setMangaList] = useState<MangaTitles[]>([]);
  const [mangaChaptersList, setMangaChaptersList] = useState<MangaChapters[]>([]);
  const [mangaVolumesList, setMangaVolumesList] = useState<MangaVolumes[]>([]);
  const [commentsList, setCommentsList] = useState<Yorumlar[]>([]);
  const [usersList, setUsersList] = useState<KullancProfilleri[]>([]);
  const [newsList, setNewsList] = useState<Haberler[]>([]);
  const [notificationsList, setNotificationsList] = useState<Notifications[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [activeTab, setActiveTab] = useState('dashboard');

  // Form states
  const [newAnime, setNewAnime] = useState({
    title: '',
    synopsis: '',
    creator: '',
    genre: '',
    releaseYear: new Date().getFullYear(),
    numberOfEpisodes: 1,
    isFeatured: false,
    videoUrl: '',
    posterImageUrl: ''
  });

  const [newManga, setNewManga] = useState({
    title: '',
    synopsis: '',
    author: '',
    genre: '',
    publicationDate: new Date().toISOString().split('T')[0],
    numberOfChapters: 1,
    isFeatured: false,
    coverImageUrl: ''
  });

  const [newMangaVolume, setNewMangaVolume] = useState({
    volumeTitle: '',
    volumeNumber: 1,
    readingLink: '',
    previewUrl: '',
    coverImage: '',
    mangaSeriesId: ''
  });

  const [newMangaChapter, setNewMangaChapter] = useState({
    chapterNumber: 1,
    chapterTitle: '',
    chapterContentUrl: '',
    releaseDate: new Date().toISOString().split('T')[0],
    isPublished: true,
    mangaReference: ''
  });

  const [newNews, setNewNews] = useState({
    title: '',
    excerpt: '',
    description: '',
    mainImage: '',
    publishDate: new Date().toISOString(),
    author: '',
    videoUrl: ''
  });

  const [newNotification, setNewNotification] = useState({
    title: '',
    content: '',
    redirectionLink: '',
    isActive: true,
    displayOrder: 1
  });

  // Edit states
  const [editingAnime, setEditingAnime] = useState<AnimeTitles | null>(null);
  const [editingManga, setEditingManga] = useState<MangaTitles | null>(null);
  const [editingMangaChapter, setEditingMangaChapter] = useState<MangaChapters | null>(null);
  const [editingMangaVolume, setEditingMangaVolume] = useState<MangaVolumes | null>(null);
  const [editingNews, setEditingNews] = useState<Haberler | null>(null);
  const [editingNotification, setEditingNotification] = useState<Notifications | null>(null);
  const [isAnimeDialogOpen, setIsAnimeDialogOpen] = useState(false);
  const [isMangaDialogOpen, setIsMangaDialogOpen] = useState(false);
  const [isMangaChapterDialogOpen, setIsMangaChapterDialogOpen] = useState(false);
  const [isMangaVolumeDialogOpen, setIsMangaVolumeDialogOpen] = useState(false);
  const [isNewsDialogOpen, setIsNewsDialogOpen] = useState(false);
  const [isNotificationDialogOpen, setIsNotificationDialogOpen] = useState(false);

  useEffect(() => {
    const fetchData = async () => {
      try {
        const [animeResponse, mangaResponse, mangaChaptersResponse, mangaVolumesResponse, commentsResponse, usersResponse, newsResponse, notificationsResponse] = await Promise.all([
          BaseCrudService.getAll<AnimeTitles>('animetitles'),
          BaseCrudService.getAll<MangaTitles>('mangatitles'),
          BaseCrudService.getAll<MangaChapters>('mangachapters'),
          BaseCrudService.getAll<MangaVolumes>('mangavolumes'),
          BaseCrudService.getAll<Yorumlar>('yorumlar'),
          BaseCrudService.getAll<KullancProfilleri>('kullancprofilleri'),
          BaseCrudService.getAll<Haberler>('haberler'),
          BaseCrudService.getAll<Notifications>('notifications')
        ]);

        setAnimeList(animeResponse.items);
        setMangaList(mangaResponse.items);
        setMangaChaptersList(mangaChaptersResponse.items);
        setMangaVolumesList(mangaVolumesResponse.items);
        setCommentsList(commentsResponse.items);
        setUsersList(usersResponse.items);
        setNewsList(newsResponse.items);
        setNotificationsList(notificationsResponse.items);
      } catch (error) {
        console.error('Veri yüklenirken hata:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchData();
  }, []);

  const handleAddAnime = async () => {
    try {
      const animeData = {
        ...newAnime,
        _id: crypto.randomUUID(),
        _createdDate: new Date(),
        _updatedDate: new Date()
      };

      await BaseCrudService.create('animetitles', animeData);
      
      // Refresh anime list
      const response = await BaseCrudService.getAll<AnimeTitles>('animetitles');
      setAnimeList(response.items);
      
      // Reset form
      setNewAnime({
        title: '',
        synopsis: '',
        creator: '',
        genre: '',
        releaseYear: new Date().getFullYear(),
        numberOfEpisodes: 1,
        isFeatured: false,
        videoUrl: '',
        posterImageUrl: ''
      });
      setIsAnimeDialogOpen(false);
    } catch (error) {
      console.error('Anime eklenirken hata:', error);
    }
  };

  const handleAddManga = async () => {
    try {
      const mangaData = {
        ...newManga,
        _id: crypto.randomUUID(),
        _createdDate: new Date(),
        _updatedDate: new Date()
      };

      await BaseCrudService.create('mangatitles', mangaData);
      
      // Refresh manga list
      const response = await BaseCrudService.getAll<MangaTitles>('mangatitles');
      setMangaList(response.items);
      
      // Reset form
      setNewManga({
        title: '',
        synopsis: '',
        author: '',
        genre: '',
        publicationDate: new Date().toISOString().split('T')[0],
        numberOfChapters: 1,
        isFeatured: false,
        coverImageUrl: ''
      });
      setIsMangaDialogOpen(false);
    } catch (error) {
      console.error('Manga eklenirken hata:', error);
    }
  };

  const handleEditAnime = async () => {
    if (!editingAnime) return;
    
    try {
      await BaseCrudService.update('animetitles', {
        ...editingAnime,
        _updatedDate: new Date()
      });
      
      // Refresh anime list
      const response = await BaseCrudService.getAll<AnimeTitles>('animetitles');
      setAnimeList(response.items);
      setEditingAnime(null);
      setIsAnimeDialogOpen(false);
    } catch (error) {
      console.error('Anime güncellenirken hata:', error);
    }
  };

  const handleEditManga = async () => {
    if (!editingManga) return;
    
    try {
      await BaseCrudService.update('mangatitles', {
        ...editingManga,
        _updatedDate: new Date()
      });
      
      // Refresh manga list
      const response = await BaseCrudService.getAll<MangaTitles>('mangatitles');
      setMangaList(response.items);
      setEditingManga(null);
      setIsMangaDialogOpen(false);
    } catch (error) {
      console.error('Manga güncellenirken hata:', error);
    }
  };

  const handleDeleteAnime = async (animeId: string) => {
    try {
      await BaseCrudService.delete('animetitles', animeId);
      
      // Refresh anime list
      const response = await BaseCrudService.getAll<AnimeTitles>('animetitles');
      setAnimeList(response.items);
    } catch (error) {
      console.error('Anime silinirken hata:', error);
    }
  };

  const handleDeleteManga = async (mangaId: string) => {
    try {
      await BaseCrudService.delete('mangatitles', mangaId);
      
      // Refresh manga list
      const response = await BaseCrudService.getAll<MangaTitles>('mangatitles');
      setMangaList(response.items);
    } catch (error) {
      console.error('Manga silinirken hata:', error);
    }
  };

  const handleAddMangaVolume = async () => {
    try {
      const volumeData = {
        ...newMangaVolume,
        _id: crypto.randomUUID(),
        _createdDate: new Date(),
        _updatedDate: new Date()
      };

      await BaseCrudService.create('mangavolumes', volumeData);
      
      // Refresh manga volumes list
      const response = await BaseCrudService.getAll<MangaVolumes>('mangavolumes');
      setMangaVolumesList(response.items);
      
      // Reset form
      setNewMangaVolume({
        volumeTitle: '',
        volumeNumber: 1,
        readingLink: '',
        previewUrl: '',
        coverImage: '',
        mangaSeriesId: ''
      });
      setIsMangaVolumeDialogOpen(false);
    } catch (error) {
      console.error('Manga cildi eklenirken hata:', error);
    }
  };

  const handleEditMangaVolume = async () => {
    if (!editingMangaVolume) return;
    
    try {
      await BaseCrudService.update('mangavolumes', {
        ...editingMangaVolume,
        _updatedDate: new Date()
      });
      
      // Refresh manga volumes list
      const response = await BaseCrudService.getAll<MangaVolumes>('mangavolumes');
      setMangaVolumesList(response.items);
      setEditingMangaVolume(null);
      setIsMangaVolumeDialogOpen(false);
    } catch (error) {
      console.error('Manga cildi güncellenirken hata:', error);
    }
  };

  const handleDeleteMangaVolume = async (volumeId: string) => {
    try {
      await BaseCrudService.delete('mangavolumes', volumeId);
      
      // Refresh manga volumes list
      const response = await BaseCrudService.getAll<MangaVolumes>('mangavolumes');
      setMangaVolumesList(response.items);
    } catch (error) {
      console.error('Manga cildi silinirken hata:', error);
    }
  };

  const handleDeleteComment = async (commentId: string) => {
    try {
      await BaseCrudService.delete('yorumlar', commentId);
      
      // Refresh comments list
      const response = await BaseCrudService.getAll<Yorumlar>('yorumlar');
      setCommentsList(response.items);
    } catch (error) {
      console.error('Yorum silinirken hata:', error);
    }
  };

  const handleAddNews = async () => {
    try {
      const newsData = {
        ...newNews,
        _id: crypto.randomUUID(),
        _createdDate: new Date(),
        _updatedDate: new Date()
      };

      await BaseCrudService.create('haberler', newsData);
      
      // Refresh news list
      const response = await BaseCrudService.getAll<Haberler>('haberler');
      setNewsList(response.items);
      
      // Reset form
      setNewNews({
        title: '',
        excerpt: '',
        description: '',
        mainImage: '',
        publishDate: new Date().toISOString(),
        author: '',
        videoUrl: ''
      });
      setIsNewsDialogOpen(false);
    } catch (error) {
      console.error('Haber eklenirken hata:', error);
    }
  };

  const handleEditNews = async () => {
    if (!editingNews) return;
    
    try {
      await BaseCrudService.update('haberler', {
        ...editingNews,
        _updatedDate: new Date()
      });
      
      // Refresh news list
      const response = await BaseCrudService.getAll<Haberler>('haberler');
      setNewsList(response.items);
      setEditingNews(null);
      setIsNewsDialogOpen(false);
    } catch (error) {
      console.error('Haber güncellenirken hata:', error);
    }
  };

  const handleDeleteNews = async (newsId: string) => {
    try {
      await BaseCrudService.delete('haberler', newsId);
      
      // Refresh news list
      const response = await BaseCrudService.getAll<Haberler>('haberler');
      setNewsList(response.items);
    } catch (error) {
      console.error('Haber silinirken hata:', error);
    }
  };

  // Notification handlers
  const handleAddNotification = async () => {
    try {
      const notificationData = {
        ...newNotification,
        _id: crypto.randomUUID(),
        _createdDate: new Date(),
        _updatedDate: new Date()
      };

      await BaseCrudService.create('notifications', notificationData);
      
      // Refresh notifications list
      const response = await BaseCrudService.getAll<Notifications>('notifications');
      setNotificationsList(response.items);
      
      // Reset form
      setNewNotification({
        title: '',
        content: '',
        redirectionLink: '',
        isActive: true,
        displayOrder: 1
      });
      setIsNotificationDialogOpen(false);
    } catch (error) {
      console.error('Bildirim eklenirken hata:', error);
    }
  };

  const handleEditNotification = async () => {
    if (!editingNotification) return;
    
    try {
      await BaseCrudService.update('notifications', {
        ...editingNotification,
        _updatedDate: new Date()
      });
      
      // Refresh notifications list
      const response = await BaseCrudService.getAll<Notifications>('notifications');
      setNotificationsList(response.items);
      setEditingNotification(null);
      setIsNotificationDialogOpen(false);
    } catch (error) {
      console.error('Bildirim güncellenirken hata:', error);
    }
  };

  const handleDeleteNotification = async (notificationId: string) => {
    try {
      await BaseCrudService.delete('notifications', notificationId);
      
      // Refresh notifications list
      const response = await BaseCrudService.getAll<Notifications>('notifications');
      setNotificationsList(response.items);
    } catch (error) {
      console.error('Bildirim silinirken hata:', error);
    }
  };

  const openNotificationEditDialog = (notification: Notifications) => {
    setEditingNotification(notification);
    setIsNotificationDialogOpen(true);
  };

  const openAnimeEditDialog = (anime: AnimeTitles) => {
    setEditingAnime(anime);
    setIsAnimeDialogOpen(true);
  };

  const openMangaEditDialog = (manga: MangaTitles) => {
    setEditingManga(manga);
    setIsMangaDialogOpen(true);
  };

  const openNewAnimeDialog = () => {
    setEditingAnime(null);
    setNewAnime({
      title: '',
      synopsis: '',
      creator: '',
      genre: '',
      releaseYear: new Date().getFullYear(),
      numberOfEpisodes: 1,
      isFeatured: false,
      videoUrl: '',
      posterImageUrl: ''
    });
    setIsAnimeDialogOpen(true);
  };

  const openNewMangaDialog = () => {
    setEditingManga(null);
    setNewManga({
      title: '',
      synopsis: '',
      author: '',
      genre: '',
      publicationDate: new Date().toISOString().split('T')[0],
      numberOfChapters: 1,
      isFeatured: false,
      coverImageUrl: ''
    });
    setIsMangaDialogOpen(true);
  };

  const openNewsEditDialog = (news: Haberler) => {
    setEditingNews(news);
    setIsNewsDialogOpen(true);
  };

  const openNewNewsDialog = () => {
    setEditingNews(null);
    setNewNews({
      title: '',
      excerpt: '',
      description: '',
      mainImage: '',
      publishDate: new Date().toISOString(),
      author: '',
      videoUrl: ''
    });
    setIsNewsDialogOpen(true);
  };

  if (isLoading) {
    return (
      <div className="min-h-screen bg-primary flex items-center justify-center">
        <div className="text-primary-foreground text-xl">Admin paneli yükleniyor...</div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-primary via-primary/95 to-secondary/20">
      {/* Modern Navigation */}
      <motion.nav 
        initial={{ y: -20, opacity: 0 }}
        animate={{ y: 0, opacity: 1 }}
        className="w-full px-6 py-4 backdrop-blur-xl bg-primary/80 border-b border-accentneongreen/20 sticky top-0 z-50"
      >
        <div className="flex items-center justify-between max-w-[120rem] mx-auto">
          <Link to="/" className="flex items-center space-x-3 group">
            <motion.div 
              whileHover={{ scale: 1.1, rotate: 5 }}
              className="w-10 h-10 bg-gradient-to-r from-accentneongreen to-accentpeach rounded-xl flex items-center justify-center shadow-lg"
            >
              <Shield className="w-6 h-6 text-primary" />
            </motion.div>
            <div>
              <span className="font-heading text-xl font-black text-primary-foreground group-hover:text-accentneongreen transition-colors">
                Mangius
              </span>
              <div className="text-xs text-accentneongreen font-medium">Admin Panel</div>
            </div>
          </Link>
          
          <div className="hidden md:flex items-center space-x-6">
            <Link to="/anime" className="font-paragraph text-primary-foreground/80 hover:text-accentneongreen transition-all duration-300 hover:scale-105">
              Anime
            </Link>
            <Link to="/manga" className="font-paragraph text-primary-foreground/80 hover:text-accentneongreen transition-all duration-300 hover:scale-105">
              Manga
            </Link>
            <Link to="/" className="font-paragraph text-primary-foreground/80 hover:text-accentneongreen transition-all duration-300 hover:scale-105">
              Ana Sayfa
            </Link>
          </div>

          <div className="flex items-center space-x-3">
            <Link to="/profile">
              <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                <Button variant="outline" className="border-accentneongreen/50 text-accentneongreen hover:bg-accentneongreen hover:text-primary transition-all duration-300">
                  <Users className="w-4 h-4 mr-2" />
                  {member?.profile?.nickname || 'Profil'}
                </Button>
              </motion.div>
            </Link>
            <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
              <Button 
                onClick={actions.logout}
                variant="outline" 
                className="border-red-400/50 text-red-400 hover:bg-red-400 hover:text-white transition-all duration-300"
              >
                <LogOut className="w-4 h-4 mr-2" />
                Çıkış
              </Button>
            </motion.div>
          </div>
        </div>
      </motion.nav>

      {/* Modern Admin Panel Content */}
      <div className="max-w-[120rem] mx-auto px-6 py-8">
        <motion.div 
          initial={{ y: 20, opacity: 0 }}
          animate={{ y: 0, opacity: 1 }}
          transition={{ delay: 0.2 }}
          className="mb-12"
        >
          <div className="text-center mb-8">
            <motion.h1 
              initial={{ scale: 0.9, opacity: 0 }}
              animate={{ scale: 1, opacity: 1 }}
              transition={{ delay: 0.3 }}
              className="font-heading text-5xl font-black text-primary-foreground mb-4"
            >
              <span className="bg-gradient-to-r from-accentneongreen to-accentpeach bg-clip-text text-transparent">
                Admin
              </span>{' '}
              <span className="text-primary-foreground">Kontrol Merkezi</span>
            </motion.h1>
            <motion.p 
              initial={{ y: 10, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.4 }}
              className="font-paragraph text-lg text-primary-foreground/70 max-w-2xl mx-auto"
            >
              İçerik yönetimi, kullanıcı analizi ve sistem kontrolü için gelişmiş yönetim paneli
            </motion.p>
          </div>
        </motion.div>

        <motion.div
          initial={{ y: 30, opacity: 0 }}
          animate={{ y: 0, opacity: 1 }}
          transition={{ delay: 0.5 }}
        >
          <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
            <TabsList className="grid w-full grid-cols-7 bg-gradient-to-r from-secondary/20 to-primary/50 backdrop-blur-sm border border-accentneongreen/20 rounded-2xl p-2 mb-8">
              <TabsTrigger 
                value="dashboard" 
                className="flex items-center space-x-2 data-[state=active]:bg-accentneongreen data-[state=active]:text-primary rounded-xl transition-all duration-300"
              >
                <BarChart3 className="w-4 h-4" />
                <span className="hidden sm:inline">Genel Bakış</span>
              </TabsTrigger>
              <TabsTrigger 
                value="anime" 
                className="flex items-center space-x-2 data-[state=active]:bg-accentneongreen data-[state=active]:text-primary rounded-xl transition-all duration-300"
              >
                <Play className="w-4 h-4" />
                <span className="hidden sm:inline">Anime</span>
              </TabsTrigger>
              <TabsTrigger 
                value="manga" 
                className="flex items-center space-x-2 data-[state=active]:bg-accentneongreen data-[state=active]:text-primary rounded-xl transition-all duration-300"
              >
                <BookOpen className="w-4 h-4" />
                <span className="hidden sm:inline">Manga</span>
              </TabsTrigger>
              <TabsTrigger 
                value="news" 
                className="flex items-center space-x-2 data-[state=active]:bg-accentneongreen data-[state=active]:text-primary rounded-xl transition-all duration-300"
              >
                <Newspaper className="w-4 h-4" />
                <span className="hidden sm:inline">Haberler</span>
              </TabsTrigger>
              <TabsTrigger 
                value="notifications" 
                className="flex items-center space-x-2 data-[state=active]:bg-accentneongreen data-[state=active]:text-primary rounded-xl transition-all duration-300"
              >
                <Bell className="w-4 h-4" />
                <span className="hidden sm:inline">Bildirimler</span>
              </TabsTrigger>
              <TabsTrigger 
                value="comments" 
                className="flex items-center space-x-2 data-[state=active]:bg-accentneongreen data-[state=active]:text-primary rounded-xl transition-all duration-300"
              >
                <MessageSquare className="w-4 h-4" />
                <span className="hidden sm:inline">Yorumlar</span>
              </TabsTrigger>
              <TabsTrigger 
                value="users" 
                className="flex items-center space-x-2 data-[state=active]:bg-accentneongreen data-[state=active]:text-primary rounded-xl transition-all duration-300"
              >
                <Users className="w-4 h-4" />
                <span className="hidden sm:inline">Kullanıcılar</span>
              </TabsTrigger>
            </TabsList>

          {/* Modern Dashboard Tab */}
          <TabsContent value="dashboard" className="space-y-8">
            <motion.div 
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.1 }}
              className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6"
            >
              <motion.div whileHover={{ scale: 1.02, y: -5 }} transition={{ duration: 0.2 }}>
                <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-accentneongreen/20 backdrop-blur-sm hover:border-accentneongreen/40 transition-all duration-300">
                  <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-3">
                    <CardTitle className="text-sm font-medium text-primary-foreground/80">
                      Toplam Anime
                    </CardTitle>
                    <div className="p-2 bg-accentneongreen/20 rounded-lg">
                      <Play className="h-5 w-5 text-accentneongreen" />
                    </div>
                  </CardHeader>
                  <CardContent>
                    <div className="text-3xl font-black text-primary-foreground mb-1">{animeList.length}</div>
                    <div className="flex items-center space-x-2">
                      <Badge className="bg-accentneongreen/20 text-accentneongreen border-accentneongreen/30">
                        <Star className="w-3 h-3 mr-1" />
                        {animeList.filter(a => a.isFeatured).length} öne çıkan
                      </Badge>
                    </div>
                  </CardContent>
                </Card>
              </motion.div>

              <motion.div whileHover={{ scale: 1.02, y: -5 }} transition={{ duration: 0.2 }}>
                <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-accentpeach/20 backdrop-blur-sm hover:border-accentpeach/40 transition-all duration-300">
                  <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-3">
                    <CardTitle className="text-sm font-medium text-primary-foreground/80">
                      Toplam Manga
                    </CardTitle>
                    <div className="p-2 bg-accentpeach/20 rounded-lg">
                      <BookOpen className="h-5 w-5 text-accentpeach" />
                    </div>
                  </CardHeader>
                  <CardContent>
                    <div className="text-3xl font-black text-primary-foreground mb-1">{mangaList.length}</div>
                    <div className="flex items-center space-x-2">
                      <Badge className="bg-accentpeach/20 text-accentpeach border-accentpeach/30">
                        <Star className="w-3 h-3 mr-1" />
                        {mangaList.filter(m => m.isFeatured).length} öne çıkan
                      </Badge>
                    </div>
                  </CardContent>
                </Card>
              </motion.div>

              <motion.div whileHover={{ scale: 1.02, y: -5 }} transition={{ duration: 0.2 }}>
                <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-blue-400/20 backdrop-blur-sm hover:border-blue-400/40 transition-all duration-300">
                  <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-3">
                    <CardTitle className="text-sm font-medium text-primary-foreground/80">
                      Toplam Yorum
                    </CardTitle>
                    <div className="p-2 bg-blue-400/20 rounded-lg">
                      <MessageSquare className="h-5 w-5 text-blue-400" />
                    </div>
                  </CardHeader>
                  <CardContent>
                    <div className="text-3xl font-black text-primary-foreground mb-1">{commentsList.length}</div>
                    <div className="flex items-center space-x-2">
                      <Badge className="bg-blue-400/20 text-blue-400 border-blue-400/30">
                        <Activity className="w-3 h-3 mr-1" />
                        Aktif yorumlar
                      </Badge>
                    </div>
                  </CardContent>
                </Card>
              </motion.div>

              <motion.div whileHover={{ scale: 1.02, y: -5 }} transition={{ duration: 0.2 }}>
                <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-purple-400/20 backdrop-blur-sm hover:border-purple-400/40 transition-all duration-300">
                  <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-3">
                    <CardTitle className="text-sm font-medium text-primary-foreground/80">
                      Kayıtlı Kullanıcı
                    </CardTitle>
                    <div className="p-2 bg-purple-400/20 rounded-lg">
                      <Users className="h-5 w-5 text-purple-400" />
                    </div>
                  </CardHeader>
                  <CardContent>
                    <div className="text-3xl font-black text-primary-foreground mb-1">{usersList.length}</div>
                    <div className="flex items-center space-x-2">
                      <Badge className="bg-purple-400/20 text-purple-400 border-purple-400/30">
                        <TrendingUp className="w-3 h-3 mr-1" />
                        Aktif profiller
                      </Badge>
                    </div>
                  </CardContent>
                </Card>
              </motion.div>
            </motion.div>

            {/* Modern Recent Activity */}
            <motion.div
              initial={{ y: 30, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.3 }}
            >
              <Card className="bg-gradient-to-br from-secondary/10 to-primary/20 border border-accentneongreen/20 backdrop-blur-sm">
                <CardHeader className="pb-4">
                  <div className="flex items-center justify-between">
                    <CardTitle className="text-xl font-bold text-primary-foreground flex items-center">
                      <Clock className="w-5 h-5 mr-2 text-accentneongreen" />
                      Son Yorumlar
                    </CardTitle>
                    <Badge className="bg-accentneongreen/20 text-accentneongreen">
                      {commentsList.length} toplam
                    </Badge>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <AnimatePresence>
                      {commentsList.slice(0, 5).map((comment, index) => (
                        <motion.div
                          key={comment._id}
                          initial={{ x: -20, opacity: 0 }}
                          animate={{ x: 0, opacity: 1 }}
                          transition={{ delay: index * 0.1 }}
                          className="flex items-start space-x-4 p-4 bg-gradient-to-r from-primary/30 to-secondary/20 rounded-xl border border-accentneongreen/10 hover:border-accentneongreen/30 transition-all duration-300"
                        >
                          <div className="p-2 bg-accentneongreen/20 rounded-lg">
                            <MessageSquare className="w-4 h-4 text-accentneongreen" />
                          </div>
                          <div className="flex-1">
                            <p className="text-sm text-primary-foreground/90 line-clamp-2 mb-2">
                              {comment.commentText}
                            </p>
                            <div className="flex items-center space-x-3">
                              <Badge variant="outline" className="text-xs border-accentneongreen/30 text-accentneongreen">
                                {comment.contentType}
                              </Badge>
                              <span className="text-xs text-primary-foreground/60 flex items-center">
                                <Calendar className="w-3 h-3 mr-1" />
                                {comment.creationDate && new Date(comment.creationDate).toLocaleDateString('tr-TR')}
                              </span>
                            </div>
                          </div>
                        </motion.div>
                      ))}
                    </AnimatePresence>
                  </div>
                </CardContent>
              </Card>
            </motion.div>
          </TabsContent>

          {/* Modern Anime Management Tab */}
          <TabsContent value="anime" className="space-y-6">
            <motion.div 
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4"
            >
              <div>
                <h2 className="font-heading text-3xl font-bold text-primary-foreground mb-2">Anime Yönetimi</h2>
                <p className="text-primary-foreground/70">Anime içeriklerini ekle, düzenle ve yönet</p>
              </div>
              <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                <Button 
                  onClick={openNewAnimeDialog} 
                  className="bg-gradient-to-r from-accentneongreen to-accentneongreen/80 text-primary hover:from-accentneongreen/90 hover:to-accentneongreen/70 shadow-lg hover:shadow-accentneongreen/25 transition-all duration-300"
                >
                  <Plus className="w-5 h-5 mr-2" />
                  Yeni Anime Ekle
                </Button>
              </motion.div>
            </motion.div>

            <motion.div 
              initial={{ y: 30, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.2 }}
              className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6"
            >
              <AnimatePresence>
                {animeList.map((anime, index) => (
                  <motion.div
                    key={anime._id}
                    initial={{ y: 20, opacity: 0 }}
                    animate={{ y: 0, opacity: 1 }}
                    transition={{ delay: index * 0.1 }}
                    whileHover={{ y: -8, scale: 1.02 }}
                    className="group"
                  >
                    <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-accentneongreen/20 backdrop-blur-sm hover:border-accentneongreen/40 transition-all duration-300 overflow-hidden">
                      <CardContent className="p-6">
                        <div className="flex justify-between items-start mb-4">
                          <h3 className="font-heading text-lg font-bold text-primary-foreground line-clamp-2 group-hover:text-accentneongreen transition-colors">
                            {anime.title}
                          </h3>
                          {anime.isFeatured && (
                            <motion.div
                              initial={{ scale: 0 }}
                              animate={{ scale: 1 }}
                              className="ml-2"
                            >
                              <Badge className="bg-gradient-to-r from-accentneongreen to-accentpeach text-primary border-0">
                                <Star className="w-3 h-3 mr-1" />
                                Öne Çıkan
                              </Badge>
                            </motion.div>
                          )}
                        </div>
                        
                        <p className="text-sm text-primary-foreground/70 line-clamp-3 mb-4">
                          {anime.synopsis}
                        </p>
                        
                        <div className="space-y-3 mb-4">
                          <div className="flex items-center justify-between text-xs text-primary-foreground/60">
                            <span className="flex items-center">
                              <Users className="w-3 h-3 mr-1" />
                              {anime.creator}
                            </span>
                            <span className="flex items-center">
                              <Calendar className="w-3 h-3 mr-1" />
                              {anime.releaseYear}
                            </span>
                          </div>
                          
                          <div className="flex items-center justify-between">
                            <Badge variant="outline" className="text-xs border-accentneongreen/30 text-accentneongreen">
                              {anime.numberOfEpisodes} Bölüm
                            </Badge>
                            {anime.videoUrl && (
                              <Badge className="bg-blue-500/20 text-blue-400 border-blue-500/30">
                                <FileVideo className="w-3 h-3 mr-1" />
                                Video
                              </Badge>
                            )}
                          </div>
                        </div>
                        
                        <div className="flex space-x-2">
                          <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }} className="flex-1">
                            <Button 
                              size="sm" 
                              variant="outline" 
                              className="w-full border-accentneongreen/50 text-accentneongreen hover:bg-accentneongreen hover:text-primary transition-all duration-300"
                              onClick={() => openAnimeEditDialog(anime)}
                            >
                              <Edit className="w-3 h-3 mr-1" />
                              Düzenle
                            </Button>
                          </motion.div>
                          <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                            <Button 
                              size="sm" 
                              variant="outline" 
                              className="border-red-400/50 text-red-400 hover:bg-red-400 hover:text-white transition-all duration-300"
                              onClick={() => handleDeleteAnime(anime._id)}
                            >
                              <Trash2 className="w-3 h-3" />
                            </Button>
                          </motion.div>
                        </div>
                      </CardContent>
                    </Card>
                  </motion.div>
                ))}
              </AnimatePresence>
            </motion.div>
          </TabsContent>

          {/* News Management Tab */}
          <TabsContent value="news" className="space-y-6">
            <motion.div 
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4"
            >
              <div>
                <h2 className="font-heading text-3xl font-bold text-primary-foreground mb-2">Haber Yönetimi</h2>
                <p className="text-primary-foreground/70">Haber içeriklerini ekle, düzenle ve yönet</p>
              </div>
              <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                <Button 
                  onClick={openNewNewsDialog} 
                  className="bg-gradient-to-r from-orange-500 to-orange-500/80 text-white hover:from-orange-500/90 hover:to-orange-500/70 shadow-lg hover:shadow-orange-500/25 transition-all duration-300"
                >
                  <Plus className="w-5 h-5 mr-2" />
                  Yeni Haber Ekle
                </Button>
              </motion.div>
            </motion.div>

            <motion.div 
              initial={{ y: 30, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.2 }}
              className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6"
            >
              <AnimatePresence>
                {newsList.map((news, index) => (
                  <motion.div
                    key={news._id}
                    initial={{ y: 20, opacity: 0 }}
                    animate={{ y: 0, opacity: 1 }}
                    transition={{ delay: index * 0.1 }}
                    whileHover={{ y: -8, scale: 1.02 }}
                    className="group"
                  >
                    <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-orange-500/20 backdrop-blur-sm hover:border-orange-500/40 transition-all duration-300 overflow-hidden">
                      <CardContent className="p-6">
                        {news.mainImage && (
                          <div className="mb-4 rounded-lg overflow-hidden">
                            <ImageComponent
                              src={news.mainImage}
                              alt={news.title || 'Haber görseli'}
                              width={300}
                              height={200}
                              className="w-full h-32 object-cover group-hover:scale-105 transition-transform duration-300"
                            />
                          </div>
                        )}
                        
                        <h3 className="font-heading text-lg font-bold text-primary-foreground line-clamp-2 group-hover:text-orange-500 transition-colors mb-2">
                          {news.title}
                        </h3>
                        
                        {news.excerpt && (
                          <p className="text-sm text-primary-foreground/70 line-clamp-2 mb-3">
                            {news.excerpt}
                          </p>
                        )}
                        
                        <div className="space-y-3 mb-4">
                          <div className="flex items-center justify-between text-xs text-primary-foreground/60">
                            {news.author && (
                              <span className="flex items-center">
                                <Users className="w-3 h-3 mr-1" />
                                {news.author}
                              </span>
                            )}
                            {news.publishDate && (
                              <span className="flex items-center">
                                <Calendar className="w-3 h-3 mr-1" />
                                {new Date(news.publishDate).toLocaleDateString('tr-TR')}
                              </span>
                            )}
                          </div>
                          
                          <div className="flex items-center justify-between">
                            <Badge className="bg-orange-500/20 text-orange-500 border-orange-500/30">
                              <Newspaper className="w-3 h-3 mr-1" />
                              Haber
                            </Badge>
                            {news.videoUrl && (
                              <VideoPlatformBadge videoUrl={news.videoUrl} />
                            )}
                          </div>
                        </div>
                        
                        <div className="flex space-x-2">
                          <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }} className="flex-1">
                            <Button 
                              size="sm" 
                              variant="outline" 
                              className="w-full border-orange-500/50 text-orange-500 hover:bg-orange-500 hover:text-white transition-all duration-300"
                              onClick={() => openNewsEditDialog(news)}
                            >
                              <Edit className="w-3 h-3 mr-1" />
                              Düzenle
                            </Button>
                          </motion.div>
                          <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                            <Button 
                              size="sm" 
                              variant="outline" 
                              className="border-red-400/50 text-red-400 hover:bg-red-400 hover:text-white transition-all duration-300"
                              onClick={() => handleDeleteNews(news._id)}
                            >
                              <Trash2 className="w-3 h-3" />
                            </Button>
                          </motion.div>
                        </div>
                      </CardContent>
                    </Card>
                  </motion.div>
                ))}
              </AnimatePresence>
            </motion.div>
          </TabsContent>

          {/* Modern Manga Management Tab */}
          <TabsContent value="manga" className="space-y-6">
            <motion.div 
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4"
            >
              <div>
                <h2 className="font-heading text-3xl font-bold text-primary-foreground mb-2">Manga Yönetimi</h2>
                <p className="text-primary-foreground/70">Manga içeriklerini ekle, düzenle ve yönet</p>
              </div>
              <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                <Button 
                  onClick={openNewMangaDialog} 
                  className="bg-gradient-to-r from-accentpeach to-accentpeach/80 text-primary hover:from-accentpeach/90 hover:to-accentpeach/70 shadow-lg hover:shadow-accentpeach/25 transition-all duration-300"
                >
                  <Plus className="w-5 h-5 mr-2" />
                  Yeni Manga Ekle
                </Button>
              </motion.div>
            </motion.div>

            <motion.div 
              initial={{ y: 30, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.2 }}
              className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6"
            >
              <AnimatePresence>
                {mangaList.map((manga, index) => (
                  <motion.div
                    key={manga._id}
                    initial={{ y: 20, opacity: 0 }}
                    animate={{ y: 0, opacity: 1 }}
                    transition={{ delay: index * 0.1 }}
                    whileHover={{ y: -8, scale: 1.02 }}
                    className="group"
                  >
                    <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-accentpeach/20 backdrop-blur-sm hover:border-accentpeach/40 transition-all duration-300 overflow-hidden">
                      <CardContent className="p-6">
                        <div className="flex justify-between items-start mb-4">
                          <h3 className="font-heading text-lg font-bold text-primary-foreground line-clamp-2 group-hover:text-accentpeach transition-colors">
                            {manga.title}
                          </h3>
                          {manga.isFeatured && (
                            <motion.div
                              initial={{ scale: 0 }}
                              animate={{ scale: 1 }}
                              className="ml-2"
                            >
                              <Badge className="bg-gradient-to-r from-accentpeach to-accentneongreen text-primary border-0">
                                <Star className="w-3 h-3 mr-1" />
                                Öne Çıkan
                              </Badge>
                            </motion.div>
                          )}
                        </div>
                        
                        <p className="text-sm text-primary-foreground/70 line-clamp-3 mb-4">
                          {manga.synopsis}
                        </p>
                        
                        <div className="space-y-3 mb-4">
                          <div className="flex items-center justify-between text-xs text-primary-foreground/60">
                            <span className="flex items-center">
                              <Users className="w-3 h-3 mr-1" />
                              {manga.author}
                            </span>
                            <span className="flex items-center">
                              <Calendar className="w-3 h-3 mr-1" />
                              {manga.publicationDate && new Date(manga.publicationDate).getFullYear()}
                            </span>
                          </div>
                          
                          <div className="flex items-center justify-between">
                            <Badge variant="outline" className="text-xs border-accentpeach/30 text-accentpeach">
                              {manga.numberOfChapters} Bölüm
                            </Badge>
                            {manga.pdfUrl && (
                              <Badge className="bg-orange-500/20 text-orange-400 border-orange-500/30">
                                <FileText className="w-3 h-3 mr-1" />
                                PDF
                              </Badge>
                            )}
                          </div>
                        </div>
                        
                        <div className="flex space-x-2">
                          <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }} className="flex-1">
                            <Button 
                              size="sm" 
                              variant="outline" 
                              className="w-full border-accentpeach/50 text-accentpeach hover:bg-accentpeach hover:text-primary transition-all duration-300"
                              onClick={() => openMangaEditDialog(manga)}
                            >
                              <Edit className="w-3 h-3 mr-1" />
                              Düzenle
                            </Button>
                          </motion.div>
                          <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                            <Button 
                              size="sm" 
                              variant="outline" 
                              className="border-red-400/50 text-red-400 hover:bg-red-400 hover:text-white transition-all duration-300"
                              onClick={() => handleDeleteManga(manga._id)}
                            >
                              <Trash2 className="w-3 h-3" />
                            </Button>
                          </motion.div>
                        </div>
                      </CardContent>
                    </Card>
                  </motion.div>
                ))}
              </AnimatePresence>
            </motion.div>
          </TabsContent>

          {/* Manga Volumes Management Tab */}
          {/* Notifications Tab */}
          <TabsContent value="notifications" className="space-y-6">
            <motion.div 
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4"
            >
              <div>
                <h2 className="font-heading text-3xl font-bold text-primary-foreground mb-2">Bildirim Yönetimi</h2>
                <p className="text-primary-foreground/70">Site bildirimlerini ekle, düzenle ve yönet</p>
              </div>
              <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                <Button 
                  onClick={() => {
                    setEditingNotification(null);
                    setNewNotification({
                      title: '',
                      content: '',
                      redirectionLink: '',
                      isActive: true,
                      displayOrder: 1
                    });
                    setIsNotificationDialogOpen(true);
                  }} 
                  className="bg-gradient-to-r from-blue-500 to-blue-500/80 text-white hover:from-blue-500/90 hover:to-blue-500/70 shadow-lg hover:shadow-blue-500/25 transition-all duration-300"
                >
                  <Plus className="w-5 h-5 mr-2" />
                  Yeni Bildirim Ekle
                </Button>
              </motion.div>
            </motion.div>

            <motion.div 
              initial={{ y: 30, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.2 }}
              className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6"
            >
              <AnimatePresence>
                {notificationsList.map((notification, index) => (
                  <motion.div
                    key={notification._id}
                    initial={{ y: 20, opacity: 0 }}
                    animate={{ y: 0, opacity: 1 }}
                    transition={{ delay: index * 0.1 }}
                    whileHover={{ y: -8, scale: 1.02 }}
                    className="group"
                  >
                    <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-blue-500/20 backdrop-blur-sm hover:border-blue-500/40 transition-all duration-300 overflow-hidden">
                      <CardContent className="p-6">
                        <div className="flex items-start justify-between mb-3">
                          <h3 className="font-heading text-lg font-bold text-primary-foreground line-clamp-2 group-hover:text-blue-500 transition-colors">
                            {notification.title}
                          </h3>
                          <Badge 
                            className={`ml-2 ${notification.isActive 
                              ? 'bg-green-500/20 text-green-400 border-green-500/30' 
                              : 'bg-red-500/20 text-red-400 border-red-500/30'
                            }`}
                          >
                            {notification.isActive ? 'Aktif' : 'Pasif'}
                          </Badge>
                        </div>
                        
                        <p className="text-sm text-primary-foreground/70 line-clamp-3 mb-3">
                          {notification.content}
                        </p>
                        
                        <div className="space-y-2 mb-4">
                          {notification.redirectionLink && (
                            <div className="flex items-center text-xs text-primary-foreground/60">
                              <Eye className="w-3 h-3 mr-1" />
                              <span className="truncate">{notification.redirectionLink}</span>
                            </div>
                          )}
                          <div className="flex items-center text-xs text-primary-foreground/60">
                            <TrendingUp className="w-3 h-3 mr-1" />
                            Sıra: {notification.displayOrder}
                          </div>
                        </div>
                        
                        <div className="flex space-x-2">
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => openNotificationEditDialog(notification)}
                            className="flex-1 border-blue-500/30 text-blue-400 hover:bg-blue-500/20 hover:border-blue-500/50"
                          >
                            <Edit className="w-3 h-3 mr-1" />
                            Düzenle
                          </Button>
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => handleDeleteNotification(notification._id)}
                            className="border-red-500/30 text-red-400 hover:bg-red-500/20 hover:border-red-500/50"
                          >
                            <Trash2 className="w-3 h-3" />
                          </Button>
                        </div>
                      </CardContent>
                    </Card>
                  </motion.div>
                ))}
              </AnimatePresence>
            </motion.div>
          </TabsContent>

          <TabsContent value="volumes" className="space-y-6">
            <motion.div 
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4"
            >
              <div>
                <h2 className="font-heading text-3xl font-black text-primary-foreground mb-2">
                  Manga <span className="text-accentneongreen">Ciltleri</span>
                </h2>
                <p className="font-paragraph text-primary-foreground/70">
                  Manga ciltlerini yönetin ve düzenleyin
                </p>
              </div>
              
              <Dialog open={isMangaVolumeDialogOpen} onOpenChange={setIsMangaVolumeDialogOpen}>
                <DialogTrigger asChild>
                  <Button 
                    className="bg-accentneongreen text-primary hover:bg-accentneongreen/90 shadow-lg"
                    onClick={() => {
                      setEditingMangaVolume(null);
                      setNewMangaVolume({
                        volumeTitle: '',
                        volumeNumber: 1,
                        readingLink: '',
                        previewUrl: '',
                        coverImage: '',
                        mangaSeriesId: ''
                      });
                    }}
                  >
                    <Plus className="w-4 h-4 mr-2" />
                    Yeni Cilt Ekle
                  </Button>
                </DialogTrigger>
                <DialogContent className="max-w-2xl bg-secondary border-0">
                  <DialogHeader>
                    <DialogTitle className="font-heading text-2xl font-bold text-secondary-foreground">
                      {editingMangaVolume ? 'Cilt Düzenle' : 'Yeni Cilt Ekle'}
                    </DialogTitle>
                  </DialogHeader>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4 py-4">
                    <div className="space-y-2">
                      <label className="font-paragraph text-sm font-medium text-secondary-foreground">Cilt Başlığı</label>
                      <Input
                        value={editingMangaVolume ? editingMangaVolume.volumeTitle || '' : newMangaVolume.volumeTitle}
                        onChange={(e) => {
                          if (editingMangaVolume) {
                            setEditingMangaVolume({ ...editingMangaVolume, volumeTitle: e.target.value });
                          } else {
                            setNewMangaVolume({ ...newMangaVolume, volumeTitle: e.target.value });
                          }
                        }}
                        className="bg-primary border-primary-foreground/20 text-primary-foreground"
                        placeholder="Cilt başlığını girin"
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="font-paragraph text-sm font-medium text-secondary-foreground">Cilt Numarası</label>
                      <Input
                        type="number"
                        value={editingMangaVolume ? editingMangaVolume.volumeNumber || 1 : newMangaVolume.volumeNumber}
                        onChange={(e) => {
                          if (editingMangaVolume) {
                            setEditingMangaVolume({ ...editingMangaVolume, volumeNumber: parseInt(e.target.value) });
                          } else {
                            setNewMangaVolume({ ...newMangaVolume, volumeNumber: parseInt(e.target.value) });
                          }
                        }}
                        className="bg-primary border-primary-foreground/20 text-primary-foreground"
                        placeholder="1"
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="font-paragraph text-sm font-medium text-secondary-foreground">Okuma Linki</label>
                      <Input
                        value={editingMangaVolume ? editingMangaVolume.readingLink || '' : newMangaVolume.readingLink}
                        onChange={(e) => {
                          if (editingMangaVolume) {
                            setEditingMangaVolume({ ...editingMangaVolume, readingLink: e.target.value });
                          } else {
                            setNewMangaVolume({ ...newMangaVolume, readingLink: e.target.value });
                          }
                        }}
                        className="bg-primary border-primary-foreground/20 text-primary-foreground"
                        placeholder="https://example.com/read"
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="font-paragraph text-sm font-medium text-secondary-foreground">Önizleme URL</label>
                      <Input
                        value={editingMangaVolume ? editingMangaVolume.previewUrl || '' : newMangaVolume.previewUrl}
                        onChange={(e) => {
                          if (editingMangaVolume) {
                            setEditingMangaVolume({ ...editingMangaVolume, previewUrl: e.target.value });
                          } else {
                            setNewMangaVolume({ ...newMangaVolume, previewUrl: e.target.value });
                          }
                        }}
                        className="bg-primary border-primary-foreground/20 text-primary-foreground"
                        placeholder="https://example.com/preview"
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="font-paragraph text-sm font-medium text-secondary-foreground">Kapak Görseli URL</label>
                      <Input
                        value={editingMangaVolume ? editingMangaVolume.coverImage || '' : newMangaVolume.coverImage}
                        onChange={(e) => {
                          if (editingMangaVolume) {
                            setEditingMangaVolume({ ...editingMangaVolume, coverImage: e.target.value });
                          } else {
                            setNewMangaVolume({ ...newMangaVolume, coverImage: e.target.value });
                          }
                        }}
                        className="bg-primary border-primary-foreground/20 text-primary-foreground"
                        placeholder="https://i.imgur.com/example.jpg"
                      />
                      {/* Image Preview */}
                      {(editingMangaVolume ? editingMangaVolume.coverImage : newMangaVolume.coverImage) && (
                        <motion.div
                          initial={{ opacity: 0, scale: 0.9 }}
                          animate={{ opacity: 1, scale: 1 }}
                          className="mt-3 p-3 bg-primary/30 rounded-lg border border-accentneongreen/20"
                        >
                          <p className="text-xs text-secondary-foreground/60 mb-2">Önizleme:</p>
                          <ImageComponent
                            src={editingMangaVolume ? editingMangaVolume.coverImage || '' : newMangaVolume.coverImage}
                            alt="Kapak önizlemesi"
                            width={120}
                            height={160}
                            className="rounded-lg border border-accentneongreen/30 object-cover"
                          />
                        </motion.div>
                      )}
                    </div>
                    <div className="space-y-2">
                      <label className="font-paragraph text-sm font-medium text-secondary-foreground">Manga Serisi ID</label>
                      <select
                        value={editingMangaVolume ? editingMangaVolume.mangaSeriesId || '' : newMangaVolume.mangaSeriesId}
                        onChange={(e) => {
                          if (editingMangaVolume) {
                            setEditingMangaVolume({ ...editingMangaVolume, mangaSeriesId: e.target.value });
                          } else {
                            setNewMangaVolume({ ...newMangaVolume, mangaSeriesId: e.target.value });
                          }
                        }}
                        className="w-full px-3 py-2 bg-primary border border-primary-foreground/20 text-primary-foreground rounded-md"
                      >
                        <option value="">Manga seçin</option>
                        {mangaList.map((manga) => (
                          <option key={manga._id} value={manga._id}>
                            {manga.title}
                          </option>
                        ))}
                      </select>
                    </div>
                  </div>
                  <div className="flex justify-end space-x-2">
                    <Button 
                      variant="outline" 
                      onClick={() => setIsMangaVolumeDialogOpen(false)}
                      className="border-primary-foreground/20 text-secondary-foreground hover:bg-primary-foreground hover:text-primary"
                    >
                      <X className="w-4 h-4 mr-2" />
                      İptal
                    </Button>
                    <Button 
                      onClick={editingMangaVolume ? handleEditMangaVolume : handleAddMangaVolume}
                      className="bg-accentneongreen text-primary hover:bg-accentneongreen/90"
                    >
                      <Save className="w-4 h-4 mr-2" />
                      {editingMangaVolume ? 'Güncelle' : 'Ekle'}
                    </Button>
                  </div>
                </DialogContent>
              </Dialog>
            </motion.div>

            <motion.div 
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.1 }}
              className="grid gap-4"
            >
              <AnimatePresence>
                {mangaVolumesList.map((volume, index) => {
                  const relatedManga = mangaList.find(manga => manga._id === volume.mangaSeriesId);
                  return (
                    <motion.div
                      key={volume._id}
                      initial={{ opacity: 0, y: 20 }}
                      animate={{ opacity: 1, y: 0 }}
                      exit={{ opacity: 0, y: -20 }}
                      transition={{ delay: index * 0.05 }}
                    >
                      <Card className="bg-gradient-to-r from-secondary/80 to-secondary border-0 hover:shadow-xl transition-all duration-300">
                        <CardContent className="p-6">
                          <div className="flex flex-col sm:flex-row justify-between items-start gap-4">
                            <div className="flex-1 space-y-2">
                              <div className="flex items-center gap-2">
                                <Badge className="bg-accentneongreen text-primary">
                                  Cilt {volume.volumeNumber}
                                </Badge>
                                <h3 className="font-heading text-lg font-bold text-secondary-foreground">
                                  {volume.volumeTitle}
                                </h3>
                              </div>
                              <p className="font-paragraph text-secondary-foreground/70">
                                <strong>Manga:</strong> {relatedManga?.title || 'Bilinmiyor'}
                              </p>
                              {volume.readingLink && (
                                <p className="font-paragraph text-secondary-foreground/70 text-sm">
                                  <strong>Okuma Linki:</strong> {volume.readingLink}
                                </p>
                              )}
                              {volume.previewUrl && (
                                <p className="font-paragraph text-secondary-foreground/70 text-sm">
                                  <strong>Önizleme:</strong> {volume.previewUrl}
                                </p>
                              )}
                            </div>
                            <div className="flex gap-2">
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => {
                                  setEditingMangaVolume(volume);
                                  setIsMangaVolumeDialogOpen(true);
                                }}
                                className="border-accentneongreen/30 text-accentneongreen hover:bg-accentneongreen hover:text-primary"
                              >
                                <Edit className="w-4 h-4" />
                              </Button>
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => handleDeleteMangaVolume(volume._id)}
                                className="border-red-500/30 text-red-500 hover:bg-red-500 hover:text-white"
                              >
                                <Trash2 className="w-4 h-4" />
                              </Button>
                            </div>
                          </div>
                        </CardContent>
                      </Card>
                    </motion.div>
                  );
                })}
              </AnimatePresence>
            </motion.div>
          </TabsContent>

          {/* Modern Comments Management Tab */}
          <TabsContent value="comments" className="space-y-6">
            <motion.div 
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4"
            >
              <div>
                <h2 className="font-heading text-3xl font-bold text-primary-foreground mb-2">Yorum Yönetimi</h2>
                <p className="text-primary-foreground/70">Kullanıcı yorumlarını incele ve yönet</p>
              </div>
              <Badge className="bg-gradient-to-r from-blue-500/20 to-purple-500/20 text-blue-400 border border-blue-500/30 px-4 py-2">
                <MessageSquare className="w-4 h-4 mr-2" />
                {commentsList.length} Toplam Yorum
              </Badge>
            </motion.div>

            <motion.div 
              initial={{ y: 30, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.2 }}
              className="space-y-4"
            >
              <AnimatePresence>
                {commentsList.map((comment, index) => (
                  <motion.div
                    key={comment._id}
                    initial={{ x: -20, opacity: 0 }}
                    animate={{ x: 0, opacity: 1 }}
                    transition={{ delay: index * 0.05 }}
                    whileHover={{ scale: 1.01 }}
                  >
                    <Card className="bg-gradient-to-r from-secondary/20 to-primary/30 border border-blue-400/20 backdrop-blur-sm hover:border-blue-400/40 transition-all duration-300">
                      <CardContent className="p-6">
                        <div className="flex justify-between items-start mb-4">
                          <div className="flex items-center space-x-3">
                            <div className="p-2 bg-blue-400/20 rounded-lg">
                              <MessageSquare className="w-4 h-4 text-blue-400" />
                            </div>
                            <div className="space-y-1">
                              <Badge variant="outline" className="text-xs border-blue-400/30 text-blue-400">
                                {comment.contentType}
                              </Badge>
                              <div className="flex items-center text-xs text-primary-foreground/60">
                                <Calendar className="w-3 h-3 mr-1" />
                                {comment.creationDate && new Date(comment.creationDate).toLocaleDateString('tr-TR')}
                              </div>
                            </div>
                          </div>
                          <motion.div whileHover={{ scale: 1.1 }} whileTap={{ scale: 0.9 }}>
                            <Button 
                              size="sm" 
                              variant="outline" 
                              className="border-red-400/50 text-red-400 hover:bg-red-400 hover:text-white transition-all duration-300"
                              onClick={() => handleDeleteComment(comment._id)}
                            >
                              <Trash2 className="w-4 h-4" />
                            </Button>
                          </motion.div>
                        </div>
                        
                        <p className="text-primary-foreground/90 mb-3 leading-relaxed">
                          {comment.commentText}
                        </p>
                        
                        <div className="flex items-center justify-between pt-3 border-t border-primary-foreground/10">
                          <span className="text-xs text-primary-foreground/60 flex items-center">
                            <Users className="w-3 h-3 mr-1" />
                            Kullanıcı ID: {comment.userId}
                          </span>
                          <Badge className="bg-primary-foreground/10 text-primary-foreground/70">
                            #{comment._id.slice(-6)}
                          </Badge>
                        </div>
                      </CardContent>
                    </Card>
                  </motion.div>
                ))}
              </AnimatePresence>
            </motion.div>
          </TabsContent>

          {/* Modern Users Management Tab */}
          <TabsContent value="users" className="space-y-6">
            <motion.div 
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4"
            >
              <div>
                <h2 className="font-heading text-3xl font-bold text-primary-foreground mb-2">Kullanıcı Yönetimi</h2>
                <p className="text-primary-foreground/70">Kayıtlı kullanıcıları görüntüle ve yönet</p>
              </div>
              <Badge className="bg-gradient-to-r from-purple-500/20 to-pink-500/20 text-purple-400 border border-purple-500/30 px-4 py-2">
                <Users className="w-4 h-4 mr-2" />
                {usersList.length} Kayıtlı Kullanıcı
              </Badge>
            </motion.div>

            <motion.div 
              initial={{ y: 30, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.2 }}
              className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6"
            >
              <AnimatePresence>
                {usersList.map((user, index) => (
                  <motion.div
                    key={user._id}
                    initial={{ y: 20, opacity: 0 }}
                    animate={{ y: 0, opacity: 1 }}
                    transition={{ delay: index * 0.1 }}
                    whileHover={{ y: -5, scale: 1.02 }}
                    className="group"
                  >
                    <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-purple-400/20 backdrop-blur-sm hover:border-purple-400/40 transition-all duration-300">
                      <CardContent className="p-6">
                        <div className="flex items-center space-x-4 mb-4">
                          <motion.div 
                            whileHover={{ scale: 1.1, rotate: 5 }}
                            className="w-14 h-14 bg-gradient-to-r from-purple-400/20 to-pink-400/20 rounded-full flex items-center justify-center border border-purple-400/30"
                          >
                            <Users className="w-7 h-7 text-purple-400" />
                          </motion.div>
                          <div className="flex-1">
                            <h3 className="font-heading text-lg font-bold text-primary-foreground group-hover:text-purple-400 transition-colors line-clamp-1">
                              {user.displayName || 'İsimsiz Kullanıcı'}
                            </h3>
                            <div className="flex items-center text-xs text-primary-foreground/60 mt-1">
                              <Calendar className="w-3 h-3 mr-1" />
                              {user.registrationDate && new Date(user.registrationDate).toLocaleDateString('tr-TR')}
                            </div>
                          </div>
                        </div>
                        
                        {user.aboutMe && (
                          <div className="mb-4">
                            <p className="text-sm text-primary-foreground/70 line-clamp-3 leading-relaxed">
                              {user.aboutMe}
                            </p>
                          </div>
                        )}
                        
                        <div className="flex items-center justify-between pt-3 border-t border-primary-foreground/10">
                          <Badge className="bg-primary-foreground/10 text-primary-foreground/70 text-xs">
                            #{user._id.slice(-6)}
                          </Badge>
                          <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                            <Button 
                              size="sm" 
                              variant="outline" 
                              className="border-purple-400/50 text-purple-400 hover:bg-purple-400 hover:text-white transition-all duration-300"
                            >
                              <Eye className="w-3 h-3 mr-1" />
                              Görüntüle
                            </Button>
                          </motion.div>
                        </div>
                      </CardContent>
                    </Card>
                  </motion.div>
                ))}
              </AnimatePresence>
            </motion.div>
          </TabsContent>
        </Tabs>
      </motion.div>
      </div>

      {/* Modern Anime Dialog */}
      <AnimatePresence>
        {isAnimeDialogOpen && (
          <Dialog open={isAnimeDialogOpen} onOpenChange={setIsAnimeDialogOpen}>
            <DialogContent className="bg-gradient-to-br from-secondary/90 to-primary/80 backdrop-blur-xl border border-accentneongreen/30 max-w-3xl max-h-[90vh] overflow-y-auto">
              <motion.div
                initial={{ scale: 0.9, opacity: 0 }}
                animate={{ scale: 1, opacity: 1 }}
                exit={{ scale: 0.9, opacity: 0 }}
                transition={{ duration: 0.2 }}
              >
                <DialogHeader className="pb-6">
                  <DialogTitle className="text-2xl font-bold text-primary-foreground flex items-center">
                    <div className="p-2 bg-accentneongreen/20 rounded-lg mr-3">
                      <Play className="w-6 h-6 text-accentneongreen" />
                    </div>
                    {editingAnime ? 'Anime Düzenle' : 'Yeni Anime Ekle'}
                  </DialogTitle>
                </DialogHeader>
                
                <div className="space-y-6">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                        <BookOpen className="w-4 h-4 mr-2 text-accentneongreen" />
                        Başlık
                      </label>
                      <Input
                        value={editingAnime ? editingAnime.title || '' : newAnime.title}
                        onChange={(e) => editingAnime 
                          ? setEditingAnime({...editingAnime, title: e.target.value})
                          : setNewAnime({...newAnime, title: e.target.value})
                        }
                        className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                        placeholder="Anime başlığını girin..."
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                        <Users className="w-4 h-4 mr-2 text-accentneongreen" />
                        Yaratıcı
                      </label>
                      <Input
                        value={editingAnime ? editingAnime.creator || '' : newAnime.creator}
                        onChange={(e) => editingAnime 
                          ? setEditingAnime({...editingAnime, creator: e.target.value})
                          : setNewAnime({...newAnime, creator: e.target.value})
                        }
                        className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                        placeholder="Yaratıcı adını girin..."
                      />
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <FileText className="w-4 h-4 mr-2 text-accentneongreen" />
                      Özet
                    </label>
                    <Textarea
                      value={editingAnime ? editingAnime.synopsis || '' : newAnime.synopsis}
                      onChange={(e) => editingAnime 
                        ? setEditingAnime({...editingAnime, synopsis: e.target.value})
                        : setNewAnime({...newAnime, synopsis: e.target.value})
                      }
                      className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300 min-h-[100px]"
                      placeholder="Anime özetini girin..."
                      rows={4}
                    />
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80">Tür</label>
                      <Input
                        value={editingAnime ? editingAnime.genre || '' : newAnime.genre}
                        onChange={(e) => editingAnime 
                          ? setEditingAnime({...editingAnime, genre: e.target.value})
                          : setNewAnime({...newAnime, genre: e.target.value})
                        }
                        className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                        placeholder="Aksiyon, Drama..."
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80">Çıkış Yılı</label>
                      <Input
                        type="number"
                        value={editingAnime ? editingAnime.releaseYear || '' : newAnime.releaseYear}
                        onChange={(e) => editingAnime 
                          ? setEditingAnime({...editingAnime, releaseYear: parseInt(e.target.value)})
                          : setNewAnime({...newAnime, releaseYear: parseInt(e.target.value)})
                        }
                        className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80">Bölüm Sayısı</label>
                      <Input
                        type="number"
                        value={editingAnime ? editingAnime.numberOfEpisodes || '' : newAnime.numberOfEpisodes}
                        onChange={(e) => editingAnime 
                          ? setEditingAnime({...editingAnime, numberOfEpisodes: parseInt(e.target.value)})
                          : setNewAnime({...newAnime, numberOfEpisodes: parseInt(e.target.value)})
                        }
                        className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                      />
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <FileVideo className="w-4 h-4 mr-2 text-accentneongreen" />
                      Video URL
                    </label>
                    <Input
                      value={editingAnime ? editingAnime.videoUrl || '' : newAnime.videoUrl}
                      onChange={(e) => editingAnime 
                        ? setEditingAnime({...editingAnime, videoUrl: e.target.value})
                        : setNewAnime({...newAnime, videoUrl: e.target.value})
                      }
                      className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                      placeholder="https://example.com/video.mp4"
                    />
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <Image className="w-4 h-4 mr-2 text-accentneongreen" />
                      Poster Görseli URL
                    </label>
                    <Input
                      value={editingAnime ? editingAnime.posterImageUrl || '' : newAnime.posterImageUrl}
                      onChange={(e) => editingAnime 
                        ? setEditingAnime({...editingAnime, posterImageUrl: e.target.value})
                        : setNewAnime({...newAnime, posterImageUrl: e.target.value})
                      }
                      className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                      placeholder="https://i.imgur.com/example.jpg"
                    />
                    {/* Image Preview */}
                    {(editingAnime ? editingAnime.posterImageUrl : newAnime.posterImageUrl) && (
                      <motion.div
                        initial={{ opacity: 0, scale: 0.9 }}
                        animate={{ opacity: 1, scale: 1 }}
                        className="mt-3 p-3 bg-primary/30 rounded-lg border border-accentneongreen/20"
                      >
                        <p className="text-xs text-primary-foreground/60 mb-2">Önizleme:</p>
                        <ImageComponent
                          src={editingAnime ? editingAnime.posterImageUrl || '' : newAnime.posterImageUrl}
                          alt="Poster önizlemesi"
                          width={120}
                          height={160}
                          className="rounded-lg border border-accentneongreen/30 object-cover"
                        />
                      </motion.div>
                    )}
                  </div>
                  
                  <div className="flex items-center space-x-3 p-4 bg-accentneongreen/10 rounded-lg border border-accentneongreen/20">
                    <input
                      type="checkbox"
                      id="animeFeature"
                      checked={editingAnime ? editingAnime.isFeatured || false : newAnime.isFeatured}
                      onChange={(e) => editingAnime 
                        ? setEditingAnime({...editingAnime, isFeatured: e.target.checked})
                        : setNewAnime({...newAnime, isFeatured: e.target.checked})
                      }
                      className="w-4 h-4 text-accentneongreen rounded focus:ring-accentneongreen"
                    />
                    <label htmlFor="animeFeature" className="text-sm text-primary-foreground flex items-center">
                      <Star className="w-4 h-4 mr-2 text-accentneongreen" />
                      Öne çıkan anime olarak işaretle
                    </label>
                  </div>
                  
                  <div className="flex space-x-3 pt-4">
                    <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }} className="flex-1">
                      <Button 
                        onClick={editingAnime ? handleEditAnime : handleAddAnime} 
                        className="w-full bg-gradient-to-r from-accentneongreen to-accentneongreen/80 text-primary hover:from-accentneongreen/90 hover:to-accentneongreen/70 shadow-lg transition-all duration-300"
                      >
                        <Save className="w-4 h-4 mr-2" />
                        {editingAnime ? 'Güncelle' : 'Ekle'}
                      </Button>
                    </motion.div>
                    <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                      <Button 
                        onClick={() => setIsAnimeDialogOpen(false)} 
                        variant="outline"
                        className="border-primary-foreground/30 text-primary-foreground hover:bg-primary-foreground/10 transition-all duration-300"
                      >
                        <X className="w-4 h-4 mr-2" />
                        İptal
                      </Button>
                    </motion.div>
                  </div>
                </div>
              </motion.div>
            </DialogContent>
          </Dialog>
        )}
      </AnimatePresence>

      {/* News Dialog */}
      <AnimatePresence>
        {isNewsDialogOpen && (
          <Dialog open={isNewsDialogOpen} onOpenChange={setIsNewsDialogOpen}>
            <DialogContent className="bg-gradient-to-br from-secondary/90 to-primary/80 backdrop-blur-xl border border-orange-500/30 max-w-3xl max-h-[90vh] overflow-y-auto">
              <motion.div
                initial={{ scale: 0.9, opacity: 0 }}
                animate={{ scale: 1, opacity: 1 }}
                exit={{ scale: 0.9, opacity: 0 }}
                transition={{ duration: 0.2 }}
              >
                <DialogHeader className="pb-6">
                  <DialogTitle className="text-2xl font-bold text-primary-foreground flex items-center">
                    <div className="p-2 bg-orange-500/20 rounded-lg mr-3">
                      <Newspaper className="w-6 h-6 text-orange-500" />
                    </div>
                    {editingNews ? 'Haber Düzenle' : 'Yeni Haber Ekle'}
                  </DialogTitle>
                </DialogHeader>
                
                <div className="space-y-6">
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <Newspaper className="w-4 h-4 mr-2 text-orange-500" />
                      Başlık
                    </label>
                    <Input
                      value={editingNews ? editingNews.title || '' : newNews.title}
                      onChange={(e) => editingNews 
                        ? setEditingNews({...editingNews, title: e.target.value})
                        : setNewNews({...newNews, title: e.target.value})
                      }
                      className="bg-primary/50 border-orange-500/30 text-primary-foreground focus:border-orange-500 transition-all duration-300"
                      placeholder="Haber başlığını girin..."
                    />
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <FileText className="w-4 h-4 mr-2 text-orange-500" />
                      Kısa Açıklama
                    </label>
                    <Textarea
                      value={editingNews ? editingNews.excerpt || '' : newNews.excerpt}
                      onChange={(e) => editingNews 
                        ? setEditingNews({...editingNews, excerpt: e.target.value})
                        : setNewNews({...newNews, excerpt: e.target.value})
                      }
                      className="bg-primary/50 border-orange-500/30 text-primary-foreground focus:border-orange-500 transition-all duration-300"
                      placeholder="Haber kartlarında gösterilecek kısa özet..."
                      rows={2}
                    />
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <FileText className="w-4 h-4 mr-2 text-orange-500" />
                      Detaylı Açıklama
                    </label>
                    <Textarea
                      value={editingNews ? editingNews.description || '' : newNews.description}
                      onChange={(e) => editingNews 
                        ? setEditingNews({...editingNews, description: e.target.value})
                        : setNewNews({...newNews, description: e.target.value})
                      }
                      className="bg-primary/50 border-orange-500/30 text-primary-foreground focus:border-orange-500 transition-all duration-300 min-h-[120px]"
                      placeholder="Haberin detaylı içeriğini girin..."
                      rows={6}
                    />
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                        <Users className="w-4 h-4 mr-2 text-orange-500" />
                        Yazar
                      </label>
                      <Input
                        value={editingNews ? editingNews.author || '' : newNews.author}
                        onChange={(e) => editingNews 
                          ? setEditingNews({...editingNews, author: e.target.value})
                          : setNewNews({...newNews, author: e.target.value})
                        }
                        className="bg-primary/50 border-orange-500/30 text-primary-foreground focus:border-orange-500 transition-all duration-300"
                        placeholder="Yazar adını girin..."
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                        <Calendar className="w-4 h-4 mr-2 text-orange-500" />
                        Yayın Tarihi
                      </label>
                      <Input
                        type="datetime-local"
                        value={editingNews ? (editingNews.publishDate ? new Date(editingNews.publishDate).toISOString().slice(0, 16) : '') : newNews.publishDate.slice(0, 16)}
                        onChange={(e) => editingNews 
                          ? setEditingNews({...editingNews, publishDate: e.target.value})
                          : setNewNews({...newNews, publishDate: e.target.value})
                        }
                        className="bg-primary/50 border-orange-500/30 text-primary-foreground focus:border-orange-500 transition-all duration-300"
                      />
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <Image className="w-4 h-4 mr-2 text-orange-500" />
                      Ana Görsel URL
                    </label>
                    <Input
                      value={editingNews ? editingNews.mainImage || '' : newNews.mainImage}
                      onChange={(e) => editingNews 
                        ? setEditingNews({...editingNews, mainImage: e.target.value})
                        : setNewNews({...newNews, mainImage: e.target.value})
                      }
                      className="bg-primary/50 border-orange-500/30 text-primary-foreground focus:border-orange-500 transition-all duration-300"
                      placeholder="https://i.imgur.com/example.jpg"
                    />
                    {/* Image Preview */}
                    {(editingNews ? editingNews.mainImage : newNews.mainImage) && (
                      <ImagePreview
                        src={editingNews ? editingNews.mainImage || '' : newNews.mainImage}
                        alt="Haber görseli önizlemesi"
                        width={300}
                        height={200}
                        borderColor="orange-500"
                      />
                    )}
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <FileVideo className="w-4 h-4 mr-2 text-orange-500" />
                      Video URL (İsteğe Bağlı)
                    </label>
                    <Input
                      value={editingNews ? editingNews.videoUrl || '' : newNews.videoUrl}
                      onChange={(e) => editingNews 
                        ? setEditingNews({...editingNews, videoUrl: e.target.value})
                        : setNewNews({...newNews, videoUrl: e.target.value})
                      }
                      className="bg-primary/50 border-orange-500/30 text-primary-foreground focus:border-orange-500 transition-all duration-300"
                      placeholder="YouTube veya Google Drive video linki"
                    />
                    <div className="text-xs text-primary-foreground/60 space-y-1">
                      <p className="font-medium">Desteklenen video platformları:</p>
                      <div className="pl-2 space-y-1">
                        <p>• <span className="text-red-400">YouTube:</span> https://www.youtube.com/watch?v=VIDEO_ID</p>
                        <p>• <span className="text-blue-400">Google Drive:</span> https://drive.google.com/file/d/FILE_ID/view</p>
                      </div>
                      <p className="text-primary-foreground/50 mt-2">
                        Video linki ekleyerek haber detayında otomatik video oynatıcı görüntülenir
                      </p>
                    </div>
                  </div>
                  
                  <div className="flex space-x-3 pt-4">
                    <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }} className="flex-1">
                      <Button 
                        onClick={editingNews ? handleEditNews : handleAddNews} 
                        className="w-full bg-gradient-to-r from-orange-500 to-orange-500/80 text-white hover:from-orange-500/90 hover:to-orange-500/70 shadow-lg transition-all duration-300"
                      >
                        <Save className="w-4 h-4 mr-2" />
                        {editingNews ? 'Güncelle' : 'Ekle'}
                      </Button>
                    </motion.div>
                    <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                      <Button 
                        onClick={() => setIsNewsDialogOpen(false)} 
                        variant="outline"
                        className="border-primary-foreground/30 text-primary-foreground hover:bg-primary-foreground/10 transition-all duration-300"
                      >
                        <X className="w-4 h-4 mr-2" />
                        İptal
                      </Button>
                    </motion.div>
                  </div>
                </div>
              </motion.div>
            </DialogContent>
          </Dialog>
        )}
      </AnimatePresence>

      {/* Modern Manga Dialog */}
      <AnimatePresence>
        {isMangaDialogOpen && (
          <Dialog open={isMangaDialogOpen} onOpenChange={setIsMangaDialogOpen}>
            <DialogContent className="bg-gradient-to-br from-secondary/90 to-primary/80 backdrop-blur-xl border border-accentpeach/30 max-w-3xl max-h-[90vh] overflow-y-auto">
              <motion.div
                initial={{ scale: 0.9, opacity: 0 }}
                animate={{ scale: 1, opacity: 1 }}
                exit={{ scale: 0.9, opacity: 0 }}
                transition={{ duration: 0.2 }}
              >
                <DialogHeader className="pb-6">
                  <DialogTitle className="text-2xl font-bold text-primary-foreground flex items-center">
                    <div className="p-2 bg-accentpeach/20 rounded-lg mr-3">
                      <BookOpen className="w-6 h-6 text-accentpeach" />
                    </div>
                    {editingManga ? 'Manga Düzenle' : 'Yeni Manga Ekle'}
                  </DialogTitle>
                </DialogHeader>
                
                <div className="space-y-6">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                        <BookOpen className="w-4 h-4 mr-2 text-accentpeach" />
                        Başlık
                      </label>
                      <Input
                        value={editingManga ? editingManga.title || '' : newManga.title}
                        onChange={(e) => editingManga 
                          ? setEditingManga({...editingManga, title: e.target.value})
                          : setNewManga({...newManga, title: e.target.value})
                        }
                        className="bg-primary/50 border-accentpeach/30 text-primary-foreground focus:border-accentpeach transition-all duration-300"
                        placeholder="Manga başlığını girin..."
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                        <Users className="w-4 h-4 mr-2 text-accentpeach" />
                        Yazar
                      </label>
                      <Input
                        value={editingManga ? editingManga.author || '' : newManga.author}
                        onChange={(e) => editingManga 
                          ? setEditingManga({...editingManga, author: e.target.value})
                          : setNewManga({...newManga, author: e.target.value})
                        }
                        className="bg-primary/50 border-accentpeach/30 text-primary-foreground focus:border-accentpeach transition-all duration-300"
                        placeholder="Yazar adını girin..."
                      />
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <FileText className="w-4 h-4 mr-2 text-accentpeach" />
                      Özet
                    </label>
                    <Textarea
                      value={editingManga ? editingManga.synopsis || '' : newManga.synopsis}
                      onChange={(e) => editingManga 
                        ? setEditingManga({...editingManga, synopsis: e.target.value})
                        : setNewManga({...newManga, synopsis: e.target.value})
                      }
                      className="bg-primary/50 border-accentpeach/30 text-primary-foreground focus:border-accentpeach transition-all duration-300 min-h-[100px]"
                      placeholder="Manga özetini girin..."
                      rows={4}
                    />
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80">Tür</label>
                      <Input
                        value={editingManga ? editingManga.genre || '' : newManga.genre}
                        onChange={(e) => editingManga 
                          ? setEditingManga({...editingManga, genre: e.target.value})
                          : setNewManga({...newManga, genre: e.target.value})
                        }
                        className="bg-primary/50 border-accentpeach/30 text-primary-foreground focus:border-accentpeach transition-all duration-300"
                        placeholder="Aksiyon, Romance..."
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80">Yayın Tarihi</label>
                      <Input
                        type="date"
                        value={editingManga ? (typeof editingManga.publicationDate === 'string' ? editingManga.publicationDate.split('T')[0] : '') : newManga.publicationDate}
                        onChange={(e) => editingManga 
                          ? setEditingManga({...editingManga, publicationDate: e.target.value})
                          : setNewManga({...newManga, publicationDate: e.target.value})
                        }
                        className="bg-primary/50 border-accentpeach/30 text-primary-foreground focus:border-accentpeach transition-all duration-300"
                      />
                    </div>
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80">Bölüm Sayısı</label>
                      <Input
                        type="number"
                        value={editingManga ? editingManga.numberOfChapters || '' : newManga.numberOfChapters}
                        onChange={(e) => editingManga 
                          ? setEditingManga({...editingManga, numberOfChapters: parseInt(e.target.value)})
                          : setNewManga({...newManga, numberOfChapters: parseInt(e.target.value)})
                        }
                        className="bg-primary/50 border-accentpeach/30 text-primary-foreground focus:border-accentpeach transition-all duration-300"
                      />
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <Image className="w-4 h-4 mr-2 text-accentpeach" />
                      Kapak Görseli URL
                    </label>
                    <Input
                      value={editingManga ? editingManga.coverImageUrl || '' : newManga.coverImageUrl}
                      onChange={(e) => editingManga 
                        ? setEditingManga({...editingManga, coverImageUrl: e.target.value})
                        : setNewManga({...newManga, coverImageUrl: e.target.value})
                      }
                      className="bg-primary/50 border-accentpeach/30 text-primary-foreground focus:border-accentpeach transition-all duration-300"
                      placeholder="https://i.imgur.com/example.jpg"
                    />
                    {/* Image Preview */}
                    {(editingManga ? editingManga.coverImageUrl : newManga.coverImageUrl) && (
                      <motion.div
                        initial={{ opacity: 0, scale: 0.9 }}
                        animate={{ opacity: 1, scale: 1 }}
                        className="mt-3 p-3 bg-primary/30 rounded-lg border border-accentpeach/20"
                      >
                        <p className="text-xs text-primary-foreground/60 mb-2">Önizleme:</p>
                        <ImageComponent
                          src={editingManga ? editingManga.coverImageUrl || '' : newManga.coverImageUrl}
                          alt="Kapak önizlemesi"
                          width={120}
                          height={160}
                          className="rounded-lg border border-accentpeach/30 object-cover"
                        />
                      </motion.div>
                    )}
                  </div>
                  
                  <div className="flex items-center space-x-3 p-4 bg-accentpeach/10 rounded-lg border border-accentpeach/20">
                    <input
                      type="checkbox"
                      id="mangaFeature"
                      checked={editingManga ? editingManga.isFeatured || false : newManga.isFeatured}
                      onChange={(e) => editingManga 
                        ? setEditingManga({...editingManga, isFeatured: e.target.checked})
                        : setNewManga({...newManga, isFeatured: e.target.checked})
                      }
                      className="w-4 h-4 text-accentpeach rounded focus:ring-accentpeach"
                    />
                    <label htmlFor="mangaFeature" className="text-sm text-primary-foreground flex items-center">
                      <Star className="w-4 h-4 mr-2 text-accentpeach" />
                      Öne çıkan manga olarak işaretle
                    </label>
                  </div>
                  
                  <div className="flex space-x-3 pt-4">
                    <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }} className="flex-1">
                      <Button 
                        onClick={editingManga ? handleEditManga : handleAddManga} 
                        className="w-full bg-gradient-to-r from-accentpeach to-accentpeach/80 text-primary hover:from-accentpeach/90 hover:to-accentpeach/70 shadow-lg transition-all duration-300"
                      >
                        <Save className="w-4 h-4 mr-2" />
                        {editingManga ? 'Güncelle' : 'Ekle'}
                      </Button>
                    </motion.div>
                    <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                      <Button 
                        onClick={() => setIsMangaDialogOpen(false)} 
                        variant="outline"
                        className="border-primary-foreground/30 text-primary-foreground hover:bg-primary-foreground/10 transition-all duration-300"
                      >
                        <X className="w-4 h-4 mr-2" />
                        İptal
                      </Button>
                    </motion.div>
                  </div>
                </div>
              </motion.div>
            </DialogContent>
          </Dialog>
        )}
      </AnimatePresence>

      {/* Modern Notification Dialog */}
      <AnimatePresence>
        {isNotificationDialogOpen && (
          <Dialog open={isNotificationDialogOpen} onOpenChange={setIsNotificationDialogOpen}>
            <DialogContent className="bg-gradient-to-br from-secondary/90 to-primary/80 backdrop-blur-xl border border-blue-500/30 max-w-2xl max-h-[90vh] overflow-y-auto">
              <motion.div
                initial={{ scale: 0.9, opacity: 0 }}
                animate={{ scale: 1, opacity: 1 }}
                exit={{ scale: 0.9, opacity: 0 }}
                transition={{ duration: 0.2 }}
              >
                <DialogHeader className="pb-6">
                  <DialogTitle className="text-2xl font-bold text-primary-foreground flex items-center">
                    <div className="p-2 bg-blue-500/20 rounded-lg mr-3">
                      <Bell className="w-6 h-6 text-blue-500" />
                    </div>
                    {editingNotification ? 'Bildirim Düzenle' : 'Yeni Bildirim Ekle'}
                  </DialogTitle>
                </DialogHeader>
                
                <div className="space-y-6">
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <Bell className="w-4 h-4 mr-2 text-blue-500" />
                      Bildirim Başlığı
                    </label>
                    <Input
                      value={editingNotification ? editingNotification.title || '' : newNotification.title}
                      onChange={(e) => editingNotification 
                        ? setEditingNotification({...editingNotification, title: e.target.value})
                        : setNewNotification({...newNotification, title: e.target.value})
                      }
                      className="bg-primary/50 border-blue-500/30 text-primary-foreground focus:border-blue-500 transition-all duration-300"
                      placeholder="Bildirim başlığını girin..."
                    />
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <FileText className="w-4 h-4 mr-2 text-blue-500" />
                      Bildirim İçeriği
                    </label>
                    <Textarea
                      value={editingNotification ? editingNotification.content || '' : newNotification.content}
                      onChange={(e) => editingNotification 
                        ? setEditingNotification({...editingNotification, content: e.target.value})
                        : setNewNotification({...newNotification, content: e.target.value})
                      }
                      className="bg-primary/50 border-blue-500/30 text-primary-foreground focus:border-blue-500 transition-all duration-300 min-h-[100px]"
                      placeholder="Bildirim içeriğini girin..."
                      rows={4}
                    />
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <Eye className="w-4 h-4 mr-2 text-blue-500" />
                      Yönlendirme Linki (İsteğe Bağlı)
                    </label>
                    <Input
                      value={editingNotification ? editingNotification.redirectionLink || '' : newNotification.redirectionLink}
                      onChange={(e) => editingNotification 
                        ? setEditingNotification({...editingNotification, redirectionLink: e.target.value})
                        : setNewNotification({...newNotification, redirectionLink: e.target.value})
                      }
                      className="bg-primary/50 border-blue-500/30 text-primary-foreground focus:border-blue-500 transition-all duration-300"
                      placeholder="https://example.com/detay"
                    />
                    <div className="text-xs text-primary-foreground/60">
                      Kullanıcılar bildirime tıkladığında yönlendirilecek link (boş bırakılabilir)
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                      <TrendingUp className="w-4 h-4 mr-2 text-blue-500" />
                      Görüntüleme Sırası
                    </label>
                    <Input
                      type="number"
                      min="1"
                      value={editingNotification ? editingNotification.displayOrder || 1 : newNotification.displayOrder}
                      onChange={(e) => editingNotification 
                        ? setEditingNotification({...editingNotification, displayOrder: parseInt(e.target.value)})
                        : setNewNotification({...newNotification, displayOrder: parseInt(e.target.value)})
                      }
                      className="bg-primary/50 border-blue-500/30 text-primary-foreground focus:border-blue-500 transition-all duration-300"
                      placeholder="1"
                    />
                    <div className="text-xs text-primary-foreground/60">
                      Düşük sayılar daha yüksek önceliğe sahiptir (1 = en yüksek öncelik)
                    </div>
                  </div>
                  
                  <div className="flex items-center space-x-3 p-4 bg-blue-500/10 rounded-lg border border-blue-500/20">
                    <input
                      type="checkbox"
                      id="notificationActive"
                      checked={editingNotification ? editingNotification.isActive || false : newNotification.isActive}
                      onChange={(e) => editingNotification 
                        ? setEditingNotification({...editingNotification, isActive: e.target.checked})
                        : setNewNotification({...newNotification, isActive: e.target.checked})
                      }
                      className="w-4 h-4 text-blue-500 rounded focus:ring-blue-500"
                    />
                    <label htmlFor="notificationActive" className="text-sm text-primary-foreground flex items-center">
                      <CheckCircle className="w-4 h-4 mr-2 text-blue-500" />
                      Bildirimi aktif olarak işaretle
                    </label>
                  </div>
                  
                  <div className="bg-blue-500/5 border border-blue-500/20 rounded-lg p-4">
                    <h4 className="text-sm font-medium text-primary-foreground mb-2 flex items-center">
                      <AlertCircle className="w-4 h-4 mr-2 text-blue-500" />
                      Bildirim Nasıl Çalışır?
                    </h4>
                    <div className="text-xs text-primary-foreground/70 space-y-1">
                      <p>• <strong>Aktif bildirimler</strong> kullanıcılara gösterilir</p>
                      <p>• <strong>Görüntüleme sırası</strong> bildirimlerin önceliğini belirler</p>
                      <p>• <strong>En düşük sıra numarası</strong> otomatik popup olarak gösterilir</p>
                      <p>• <strong>Yönlendirme linki</strong> varsa tıklanabilir hale gelir</p>
                    </div>
                  </div>
                  
                  <div className="flex space-x-3 pt-4">
                    <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }} className="flex-1">
                      <Button 
                        onClick={editingNotification ? handleEditNotification : handleAddNotification} 
                        className="w-full bg-gradient-to-r from-blue-500 to-blue-500/80 text-white hover:from-blue-500/90 hover:to-blue-500/70 shadow-lg transition-all duration-300"
                      >
                        <Save className="w-4 h-4 mr-2" />
                        {editingNotification ? 'Güncelle' : 'Ekle'}
                      </Button>
                    </motion.div>
                    <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                      <Button 
                        onClick={() => setIsNotificationDialogOpen(false)} 
                        variant="outline"
                        className="border-primary-foreground/30 text-primary-foreground hover:bg-primary-foreground/10 transition-all duration-300"
                      >
                        <X className="w-4 h-4 mr-2" />
                        İptal
                      </Button>
                    </motion.div>
                  </div>
                </div>
              </motion.div>
            </DialogContent>
          </Dialog>
        )}
      </AnimatePresence>
    </div>
  );
}