import { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import { motion, AnimatePresence } from 'framer-motion';
import { BaseCrudService } from '@/integrations';
import { useMember } from '@/integrations';
import { AnimeTitles, AnimeEpisodes } from '@/entities';
import { useFavorites } from '@/hooks/useFavorites';
import { useAdminAccess } from '@/hooks/useAdminAccess';
import { CommentSection } from '@/components/ui/comment-section';
import { EnhancedVideoPlayer } from '@/components/ui/enhanced-video-player';
import { ServerVideoStatus } from '@/components/ui/server-video-status';
import { Image } from '@/components/ui/image';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { 
  BookOpen, 
  ArrowLeft, 
  Play, 
  Calendar, 
  User, 
  Tag, 
  Film, 
  LogOut, 
  Heart, 
  Settings,
  Plus,
  Edit,
  Trash2,
  Clock,
  Video,
  Pause,
  Volume2,
  VolumeX,
  Maximize,
  SkipBack,
  SkipForward,
  Save,
  X,
  Youtube,
  ExternalLink,
  Upload,
  Server
} from 'lucide-react';
import { AnimeEpisodeUploadDialog } from '@/components/ui/anime-episode-upload-dialog';

export default function AnimeDetailPage() {
  const { id } = useParams<{ id: string }>();
  const [anime, setAnime] = useState<AnimeTitles | null>(null);
  const [episodes, setEpisodes] = useState<AnimeEpisodes[]>([]);
  const [relatedAnime, setRelatedAnime] = useState<AnimeTitles[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [activeTab, setActiveTab] = useState('overview');
  const [selectedEpisode, setSelectedEpisode] = useState<AnimeEpisodes | null>(null);
  const [isVideoPlaying, setIsVideoPlaying] = useState(false);
  const [isVideoMuted, setIsVideoMuted] = useState(false);
  const [videoProgress, setVideoProgress] = useState(0);
  const [videoDuration, setVideoDuration] = useState(0);
  const [currentTime, setCurrentTime] = useState(0);
  const [isEpisodeDialogOpen, setIsEpisodeDialogOpen] = useState(false);
  const [isEpisodeUploadDialogOpen, setIsEpisodeUploadDialogOpen] = useState(false);
  const [editingEpisode, setEditingEpisode] = useState<AnimeEpisodes | null>(null);
  const [newEpisode, setNewEpisode] = useState({
    episodeNumber: 1,
    episodeTitle: '',
    episodeDescription: '',
    videoUrl: '',
    duration: 0
  });
  
  const { member, isAuthenticated, actions } = useMember();
  
  // Admin access check
  const { isAdmin } = useAdminAccess();
  
  // Favorites hook
  const { addToFavorites, removeFromFavorites, isFavorite } = useFavorites(
    member?.loginEmail
  );

  const handleFavoriteToggle = async () => {
    if (!isAuthenticated || !anime) {
      actions.login();
      return;
    }

    const isCurrentlyFavorite = isFavorite(anime._id, 'anime');
    
    if (isCurrentlyFavorite) {
      await removeFromFavorites(anime._id, 'anime');
    } else {
      await addToFavorites(anime._id, 'anime');
    }
  };

  // Episode management functions
  const openNewEpisodeDialog = () => {
    setEditingEpisode(null);
    setNewEpisode({
      episodeNumber: episodes.length + 1,
      episodeTitle: '',
      episodeDescription: '',
      videoUrl: '',
      duration: 0
    });
    setIsEpisodeDialogOpen(true);
  };

  const openEpisodeEditDialog = (episode: AnimeEpisodes) => {
    setEditingEpisode(episode);
    setIsEpisodeDialogOpen(true);
  };

  const handleAddEpisode = async () => {
    try {
      const episodeData = {
        ...newEpisode,
        animeId: id, // Link episode to current anime
        _id: crypto.randomUUID()
      };
      
      await BaseCrudService.create('animeepisodes', episodeData);
      await fetchEpisodes();
      setIsEpisodeDialogOpen(false);
      setNewEpisode({
        episodeNumber: episodes.length + 2,
        episodeTitle: '',
        episodeDescription: '',
        videoUrl: '',
        duration: 0
      });
    } catch (error) {
      console.error('Error adding episode:', error);
    }
  };

  const handleEditEpisode = async () => {
    if (!editingEpisode) return;
    
    try {
      await BaseCrudService.update('animeepisodes', editingEpisode);
      await fetchEpisodes();
      setIsEpisodeDialogOpen(false);
      setEditingEpisode(null);
    } catch (error) {
      console.error('Error updating episode:', error);
    }
  };

  const handleDeleteEpisode = async (episodeId: string) => {
    if (!confirm('Bu bölümü silmek istediğinizden emin misiniz?')) return;
    
    try {
      await BaseCrudService.delete('animeepisodes', episodeId);
      await fetchEpisodes();
      if (selectedEpisode?._id === episodeId) {
        setSelectedEpisode(null);
      }
    } catch (error) {
      console.error('Error deleting episode:', error);
    }
  };

  const fetchEpisodes = async () => {
    if (!id) return;
    
    try {
      const episodesResponse = await BaseCrudService.getAll<AnimeEpisodes>('animeepisodes');
      // Filter episodes by animeId to show only episodes belonging to this anime
      const animeEpisodes = episodesResponse.items
        .filter(episode => episode.animeId === id)
        .sort((a, b) => (a.episodeNumber || 0) - (b.episodeNumber || 0));
      setEpisodes(animeEpisodes);
      
      // Auto-select first episode if none selected
      if (animeEpisodes.length > 0 && !selectedEpisode) {
        setSelectedEpisode(animeEpisodes[0]);
      }
    } catch (error) {
      console.error('Error fetching episodes:', error);
    }
  };

  const handleEpisodeAdded = () => {
    // Refresh episodes list after new episode is added
    fetchEpisodes();
  };

  // YouTube video helper functions
  const extractYouTubeVideoId = (url: string): string | null => {
    const regex = /(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/;
    const match = url.match(regex);
    return match ? match[1] : null;
  };

  const isYouTubeUrl = (url: string): boolean => {
    return url.includes('youtube.com') || url.includes('youtu.be');
  };

  const getYouTubeEmbedUrl = (url: string): string => {
    const videoId = extractYouTubeVideoId(url);
    return videoId ? `https://www.youtube.com/embed/${videoId}?autoplay=0&rel=0&modestbranding=1` : '';
  };

  const formatTime = (seconds: number) => {
    const mins = Math.floor(seconds / 60);
    const secs = Math.floor(seconds % 60);
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  // Video player helper functions
  const handleVideoLoad = () => {
    const loadingOverlay = document.getElementById('video-loading');
    if (loadingOverlay) {
      loadingOverlay.style.display = 'none';
    }
  };

  const handleVideoError = (error: any) => {
    console.error('Video playback error:', error);
    const loadingOverlay = document.getElementById('video-loading');
    if (loadingOverlay) {
      loadingOverlay.innerHTML = `
        <div class="text-white text-center">
          <div class="text-red-400 mb-2">⚠️</div>
          <p class="font-paragraph text-sm">Video yüklenemedi</p>
          <p class="font-paragraph text-xs text-white/60 mt-1">Dosya formatı desteklenmiyor olabilir</p>
        </div>
      `;
    }
  };

  useEffect(() => {
    const fetchAnime = async () => {
      if (!id) return;
      
      try {
        const animeData = await BaseCrudService.getById<AnimeTitles>('animetitles', id);
        setAnime(animeData);

        // Fetch episodes
        await fetchEpisodes();

        // Fetch related anime (same genre)
        if (animeData?.genre) {
          const allAnimeResponse = await BaseCrudService.getAll<AnimeTitles>('animetitles');
          const related = allAnimeResponse.items
            .filter(item => item.genre === animeData.genre && item._id !== animeData._id)
            .slice(0, 4);
          setRelatedAnime(related);
        }
      } catch (error) {
        console.error('Error fetching anime:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchAnime();
  }, [id]);

  if (isLoading) {
    return (
      <div className="min-h-screen bg-primary flex items-center justify-center">
        <div className="text-primary-foreground text-xl">Anime detayları yükleniyor...</div>
      </div>
    );
  }

  if (!anime) {
    return (
      <div className="min-h-screen bg-primary flex items-center justify-center">
        <div className="text-center">
          <h2 className="font-heading text-2xl font-bold text-primary-foreground mb-4">
            Anime bulunamadı
          </h2>
          <Link to="/anime">
            <Button className="bg-accentneongreen text-primary hover:bg-accentneongreen/90">
              Anime Koleksiyonuna Dön
            </Button>
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-primary via-primary/95 to-secondary/20">
      {/* Modern Navigation */}
      <motion.nav 
        initial={{ y: -20, opacity: 0 }}
        animate={{ y: 0, opacity: 1 }}
        className="w-full px-6 py-4 backdrop-blur-xl bg-primary/80 border-b border-accentneongreen/20 sticky top-0 z-50"
      >
        <div className="flex items-center justify-between max-w-[120rem] mx-auto">
          <Link to="/" className="flex items-center space-x-3 group">
            <motion.div 
              whileHover={{ scale: 1.1, rotate: 5 }}
              className="w-10 h-10 bg-gradient-to-r from-accentneongreen to-accentpeach rounded-xl flex items-center justify-center shadow-lg"
            >
              <BookOpen className="w-6 h-6 text-primary" />
            </motion.div>
            <span className="font-heading text-xl font-black text-primary-foreground group-hover:text-accentneongreen transition-colors">
              Mangius
            </span>
          </Link>
          
          <div className="hidden md:flex items-center space-x-6">
            <Link to="/anime" className="font-paragraph text-accentneongreen font-medium hover:scale-105 transition-all duration-300">
              Anime
            </Link>
            <Link to="/manga" className="font-paragraph text-primary-foreground/80 hover:text-accentneongreen transition-all duration-300 hover:scale-105">
              Manga
            </Link>
            <Link to="/" className="font-paragraph text-primary-foreground/80 hover:text-accentneongreen transition-all duration-300 hover:scale-105">
              Ana Sayfa
            </Link>
          </div>

          <div className="flex items-center space-x-3">
            {!isAuthenticated ? (
              <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                <Button 
                  onClick={actions.login}
                  variant="outline" 
                  className="border-accentneongreen/50 text-accentneongreen hover:bg-accentneongreen hover:text-primary transition-all duration-300"
                >
                  Giriş Yap
                </Button>
              </motion.div>
            ) : (
              <>
                {isAdmin && (
                  <Link to="/admin">
                    <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                      <Button variant="outline" className="border-secondary/50 text-secondary hover:bg-secondary hover:text-secondary-foreground transition-all duration-300">
                        <Settings className="w-4 h-4 mr-2" />
                        Admin
                      </Button>
                    </motion.div>
                  </Link>
                )}
                <Link to="/profile">
                  <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                    <Button variant="outline" className="border-accentneongreen/50 text-accentneongreen hover:bg-accentneongreen hover:text-primary transition-all duration-300">
                      <User className="w-4 h-4 mr-2" />
                      {member?.profile?.nickname || 'Profil'}
                    </Button>
                  </motion.div>
                </Link>
                <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                  <Button 
                    onClick={actions.logout}
                    variant="outline" 
                    className="border-red-400/50 text-red-400 hover:bg-red-400 hover:text-white transition-all duration-300"
                  >
                    <LogOut className="w-4 h-4 mr-2" />
                    Çıkış
                  </Button>
                </motion.div>
              </>
            )}
          </div>
        </div>
      </motion.nav>

      {/* Back Button */}
      <motion.div 
        initial={{ x: -20, opacity: 0 }}
        animate={{ x: 0, opacity: 1 }}
        transition={{ delay: 0.2 }}
        className="w-full max-w-[120rem] mx-auto px-6 py-6"
      >
        <Link to="/anime">
          <motion.div whileHover={{ scale: 1.02, x: -5 }} whileTap={{ scale: 0.98 }}>
            <Button variant="outline" className="border-primary-foreground/30 text-primary-foreground hover:bg-primary-foreground/10 hover:border-accentneongreen/50 transition-all duration-300">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Anime Koleksiyonuna Dön
            </Button>
          </motion.div>
        </Link>
      </motion.div>

      {/* Main Content */}
      <div className="w-full max-w-[120rem] mx-auto px-6 pb-16">
        <motion.div
          initial={{ y: 30, opacity: 0 }}
          animate={{ y: 0, opacity: 1 }}
          transition={{ delay: 0.3 }}
        >
          <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
            <TabsList className="grid w-full grid-cols-3 bg-gradient-to-r from-secondary/20 to-primary/50 backdrop-blur-sm border border-accentneongreen/20 rounded-2xl p-2 mb-8">
              <TabsTrigger 
                value="overview" 
                className="flex items-center space-x-2 data-[state=active]:bg-accentneongreen data-[state=active]:text-primary rounded-xl transition-all duration-300"
              >
                <Film className="w-4 h-4" />
                <span>Genel Bakış</span>
              </TabsTrigger>
              <TabsTrigger 
                value="episodes" 
                className="flex items-center space-x-2 data-[state=active]:bg-accentneongreen data-[state=active]:text-primary rounded-xl transition-all duration-300"
              >
                <Play className="w-4 h-4" />
                <span>Bölümler ({episodes.length})</span>
              </TabsTrigger>
              <TabsTrigger 
                value="watch" 
                className="flex items-center space-x-2 data-[state=active]:bg-accentneongreen data-[state=active]:text-primary rounded-xl transition-all duration-300"
              >
                <Video className="w-4 h-4" />
                <span>İzle</span>
              </TabsTrigger>
            </TabsList>
            {/* Overview Tab */}
            <TabsContent value="overview" className="space-y-8">
              <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
                {/* Poster and Quick Info */}
                <motion.div 
                  initial={{ x: -30, opacity: 0 }}
                  animate={{ x: 0, opacity: 1 }}
                  transition={{ delay: 0.4 }}
                  className="lg:col-span-1"
                >
                  <div className="sticky top-8">
                    <div className="aspect-[3/4] relative overflow-hidden rounded-2xl mb-6 group">
                      {(anime.posterImageUrl || anime.posterImage) ? (
                        <Image
                          src={anime.posterImageUrl || anime.posterImage || ''}
                          alt={anime.title || 'Anime posteri'}
                          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-500"
                          width={400}
                        />
                      ) : (
                        <div className="w-full h-full bg-gradient-to-br from-secondary/50 to-primary/30 flex items-center justify-center rounded-2xl">
                          <Play className="w-16 h-16 text-secondary-foreground/40" />
                        </div>
                      )}
                      
                      {anime.isFeatured && (
                        <motion.div 
                          initial={{ scale: 0, rotate: -10 }}
                          animate={{ scale: 1, rotate: 0 }}
                          className="absolute top-4 left-4"
                        >
                          <Badge className="bg-gradient-to-r from-accentneongreen to-accentpeach text-primary border-0 shadow-lg">
                            ⭐ Öne Çıkan
                          </Badge>
                        </motion.div>
                      )}
                    </div>

                    <div className="space-y-4">
                      <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                        <Button 
                          size="lg" 
                          className="w-full bg-gradient-to-r from-accentneongreen to-accentneongreen/80 text-primary hover:from-accentneongreen/90 hover:to-accentneongreen/70 shadow-lg hover:shadow-accentneongreen/25 transition-all duration-300"
                          onClick={() => setActiveTab('watch')}
                        >
                          <Play className="w-5 h-5 mr-2" />
                          Şimdi İzle
                        </Button>
                      </motion.div>
                      
                      <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                        <Button 
                          size="lg" 
                          variant="outline" 
                          className={`w-full transition-all duration-300 ${
                            anime && isFavorite(anime._id, 'anime')
                              ? 'border-red-400/50 text-red-400 hover:bg-red-400 hover:text-white'
                              : 'border-primary-foreground/30 text-primary-foreground hover:bg-primary-foreground/10 hover:border-accentneongreen/50'
                          }`}
                          onClick={handleFavoriteToggle}
                        >
                          <Heart className={`w-5 h-5 mr-2 ${anime && isFavorite(anime._id, 'anime') ? 'fill-current' : ''}`} />
                          {anime && isFavorite(anime._id, 'anime') ? 'Favorilerden Çıkar' : 'Favorilere Ekle'}
                        </Button>
                      </motion.div>
                    </div>
                  </div>
                </motion.div>

                {/* Main Content */}
                <motion.div 
                  initial={{ x: 30, opacity: 0 }}
                  animate={{ x: 0, opacity: 1 }}
                  transition={{ delay: 0.5 }}
                  className="lg:col-span-2 space-y-8"
                >
                  {/* Title and Basic Info */}
                  <div>
                    <motion.h1 
                      initial={{ y: 20, opacity: 0 }}
                      animate={{ y: 0, opacity: 1 }}
                      transition={{ delay: 0.6 }}
                      className="font-heading text-4xl md:text-5xl font-black text-primary-foreground mb-4"
                    >
                      {anime.title}
                    </motion.h1>
                    
                    <motion.div 
                      initial={{ y: 20, opacity: 0 }}
                      animate={{ y: 0, opacity: 1 }}
                      transition={{ delay: 0.7 }}
                      className="flex flex-wrap gap-4 mb-6"
                    >
                      {anime.genre && (
                        <div className="flex items-center space-x-2 bg-accentneongreen/20 px-3 py-1 rounded-full">
                          <Tag className="w-4 h-4 text-accentneongreen" />
                          <span className="font-paragraph text-accentneongreen font-medium">
                            {anime.genre}
                          </span>
                        </div>
                      )}
                      
                      {anime.releaseYear && (
                        <div className="flex items-center space-x-2 bg-primary-foreground/10 px-3 py-1 rounded-full">
                          <Calendar className="w-4 h-4 text-primary-foreground/60" />
                          <span className="font-paragraph text-primary-foreground/60">
                            {anime.releaseYear}
                          </span>
                        </div>
                      )}
                      
                      {anime.numberOfEpisodes && (
                        <div className="flex items-center space-x-2 bg-primary-foreground/10 px-3 py-1 rounded-full">
                          <Film className="w-4 h-4 text-primary-foreground/60" />
                          <span className="font-paragraph text-primary-foreground/60">
                            {anime.numberOfEpisodes} Bölüm
                          </span>
                        </div>
                      )}
                      
                      {anime.creator && (
                        <div className="flex items-center space-x-2 bg-primary-foreground/10 px-3 py-1 rounded-full">
                          <User className="w-4 h-4 text-primary-foreground/60" />
                          <span className="font-paragraph text-primary-foreground/60">
                            {anime.creator}
                          </span>
                        </div>
                      )}
                    </motion.div>
                  </div>

                  {/* Synopsis */}
                  <motion.div
                    initial={{ y: 30, opacity: 0 }}
                    animate={{ y: 0, opacity: 1 }}
                    transition={{ delay: 0.8 }}
                    className="bg-gradient-to-br from-secondary/20 to-primary/30 backdrop-blur-sm rounded-2xl p-8 border border-accentneongreen/20"
                  >
                    <h2 className="font-heading text-2xl font-bold text-primary-foreground mb-4 flex items-center">
                      <BookOpen className="w-6 h-6 mr-3 text-accentneongreen" />
                      Özet
                    </h2>
                    <p className="font-paragraph text-primary-foreground/80 leading-relaxed text-lg">
                      {anime.synopsis || 'Bu anime için özet mevcut değil.'}
                    </p>
                  </motion.div>

                  {/* Additional Info */}
                  <motion.div
                    initial={{ y: 30, opacity: 0 }}
                    animate={{ y: 0, opacity: 1 }}
                    transition={{ delay: 0.9 }}
                    className="grid grid-cols-1 md:grid-cols-2 gap-6"
                  >
                    <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-accentneongreen/20 backdrop-blur-sm">
                      <CardHeader>
                        <CardTitle className="text-primary-foreground flex items-center">
                          <Film className="w-5 h-5 mr-2 text-accentneongreen" />
                          Detaylar
                        </CardTitle>
                      </CardHeader>
                      <CardContent className="space-y-3">
                        <div className="flex justify-between">
                          <span className="font-paragraph text-primary-foreground/70">Durum:</span>
                          <span className="font-paragraph text-primary-foreground font-medium">
                            {anime.isFeatured ? 'Öne Çıkan' : 'Mevcut'}
                          </span>
                        </div>
                        {anime.releaseYear && (
                          <div className="flex justify-between">
                            <span className="font-paragraph text-primary-foreground/70">Yıl:</span>
                            <span className="font-paragraph text-primary-foreground font-medium">
                              {anime.releaseYear}
                            </span>
                          </div>
                        )}
                        {anime.numberOfEpisodes && (
                          <div className="flex justify-between">
                            <span className="font-paragraph text-primary-foreground/70">Bölümler:</span>
                            <span className="font-paragraph text-primary-foreground font-medium">
                              {anime.numberOfEpisodes}
                            </span>
                          </div>
                        )}
                      </CardContent>
                    </Card>

                    <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-accentpeach/20 backdrop-blur-sm">
                      <CardHeader>
                        <CardTitle className="text-primary-foreground flex items-center">
                          <User className="w-5 h-5 mr-2 text-accentpeach" />
                          Yapım
                        </CardTitle>
                      </CardHeader>
                      <CardContent className="space-y-3">
                        {anime.creator && (
                          <div className="flex justify-between">
                            <span className="font-paragraph text-primary-foreground/70">Yaratıcı:</span>
                            <span className="font-paragraph text-primary-foreground font-medium">
                              {anime.creator}
                            </span>
                          </div>
                        )}
                        {anime.genre && (
                          <div className="flex justify-between">
                            <span className="font-paragraph text-primary-foreground/70">Tür:</span>
                            <span className="font-paragraph text-accentpeach font-medium">
                              {anime.genre}
                            </span>
                          </div>
                        )}
                      </CardContent>
                    </Card>
                  </motion.div>

                  {/* Comments Section */}
                  <motion.div
                    initial={{ y: 30, opacity: 0 }}
                    animate={{ y: 0, opacity: 1 }}
                    transition={{ delay: 1.0 }}
                  >
                    {anime && (
                      <CommentSection 
                        contentId={anime._id} 
                        contentType="anime" 
                      />
                    )}
                  </motion.div>
                </motion.div>
              </div>
            </TabsContent>

            {/* Episodes Tab */}
            <TabsContent value="episodes" className="space-y-6">
              <motion.div
                initial={{ y: 20, opacity: 0 }}
                animate={{ y: 0, opacity: 1 }}
                className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4"
              >
                <div>
                  <h2 className="font-heading text-3xl font-bold text-primary-foreground mb-2">
                    Bölümler
                  </h2>
                  <p className="text-primary-foreground/70">
                    {episodes.length} bölüm mevcut
                  </p>
                </div>
                {isAdmin && (
                  <div className="flex gap-2">
                    <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                      <Button 
                        onClick={openNewEpisodeDialog}
                        variant="outline"
                        className="border-secondary-foreground/30 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
                      >
                        <Edit className="w-4 h-4 mr-2" />
                        Manuel Ekle
                      </Button>
                    </motion.div>
                    <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                      <Button 
                        onClick={() => setIsEpisodeUploadDialogOpen(true)}
                        className="bg-gradient-to-r from-accentneongreen to-accentneongreen/80 text-primary hover:from-accentneongreen/90 hover:to-accentneongreen/70 shadow-lg hover:shadow-accentneongreen/25 transition-all duration-300"
                      >
                        <Upload className="w-5 h-5 mr-2" />
                        Video Yükle
                      </Button>
                    </motion.div>
                  </div>
                )}
              </motion.div>

              <motion.div 
                initial={{ y: 30, opacity: 0 }}
                animate={{ y: 0, opacity: 1 }}
                transition={{ delay: 0.2 }}
                className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6"
              >
                <AnimatePresence>
                  {episodes.map((episode, index) => (
                    <motion.div
                      key={episode._id}
                      initial={{ y: 20, opacity: 0 }}
                      animate={{ y: 0, opacity: 1 }}
                      transition={{ delay: index * 0.1 }}
                      whileHover={{ y: -5, scale: 1.02 }}
                      className="group"
                    >
                      <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-accentneongreen/20 backdrop-blur-sm hover:border-accentneongreen/40 transition-all duration-300 overflow-hidden">
                        <CardContent className="p-6">
                          <div className="flex justify-between items-start mb-4">
                            <div className="flex items-center space-x-3">
                              <div className="w-12 h-12 bg-accentneongreen/20 rounded-lg flex items-center justify-center">
                                <span className="font-heading text-lg font-bold text-accentneongreen">
                                  {episode.episodeNumber}
                                </span>
                              </div>
                              <div>
                                <h3 className="font-heading text-lg font-bold text-primary-foreground line-clamp-1 group-hover:text-accentneongreen transition-colors">
                                  {episode.episodeTitle}
                                </h3>
                                {episode.duration && (
                                  <div className="flex items-center text-xs text-primary-foreground/60 mt-1">
                                    <Clock className="w-3 h-3 mr-1" />
                                    {episode.duration} dk
                                  </div>
                                )}
                              </div>
                            </div>
                          </div>
                          
                          {episode.episodeDescription && (
                            <p className="text-sm text-primary-foreground/70 line-clamp-3 mb-4 leading-relaxed">
                              {episode.episodeDescription}
                            </p>
                          )}
                          
                          {/* Video Type Indicator & Server Status */}
                          {episode.videoUrl && (
                            <div className="mb-4 space-y-2">
                              {isYouTubeUrl(episode.videoUrl) ? (
                                <Badge className="bg-red-600/20 text-red-400 border-red-400/30 flex items-center space-x-1 w-fit">
                                  <Youtube className="w-3 h-3" />
                                  <span>YouTube Video</span>
                                </Badge>
                              ) : (
                                <div className="space-y-2">
                                  <Badge className="bg-accentneongreen/20 text-accentneongreen border-accentneongreen/30 flex items-center space-x-1 w-fit">
                                    <Server className="w-3 h-3" />
                                    <span>Sunucu Videosu</span>
                                  </Badge>
                                  <ServerVideoStatus 
                                    videoUrl={episode.videoUrl} 
                                    className="text-xs"
                                  />
                                </div>
                              )}
                            </div>
                          )}
                          
                          <div className="flex space-x-2">
                            <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }} className="flex-1">
                              <Button 
                                size="sm" 
                                className="w-full bg-gradient-to-r from-accentneongreen/80 to-accentneongreen/60 text-primary hover:from-accentneongreen hover:to-accentneongreen/80 transition-all duration-300"
                                onClick={() => {
                                  setSelectedEpisode(episode);
                                  setActiveTab('watch');
                                }}
                              >
                                <Play className="w-3 h-3 mr-1" />
                                İzle
                              </Button>
                            </motion.div>
                            
                            {isAdmin && (
                              <>
                                <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                                  <Button 
                                    size="sm" 
                                    variant="outline" 
                                    className="border-blue-400/50 text-blue-400 hover:bg-blue-400 hover:text-white transition-all duration-300"
                                    onClick={() => openEpisodeEditDialog(episode)}
                                  >
                                    <Edit className="w-3 h-3" />
                                  </Button>
                                </motion.div>
                                <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                                  <Button 
                                    size="sm" 
                                    variant="outline" 
                                    className="border-red-400/50 text-red-400 hover:bg-red-400 hover:text-white transition-all duration-300"
                                    onClick={() => handleDeleteEpisode(episode._id)}
                                  >
                                    <Trash2 className="w-3 h-3" />
                                  </Button>
                                </motion.div>
                              </>
                            )}
                          </div>
                        </CardContent>
                      </Card>
                    </motion.div>
                  ))}
                </AnimatePresence>
              </motion.div>

              {episodes.length === 0 && (
                <motion.div
                  initial={{ opacity: 0 }}
                  animate={{ opacity: 1 }}
                  className="text-center py-16"
                >
                  <div className="w-24 h-24 bg-accentneongreen/20 rounded-full flex items-center justify-center mx-auto mb-6">
                    <Play className="w-12 h-12 text-accentneongreen" />
                  </div>
                  <h3 className="font-heading text-2xl font-bold text-primary-foreground mb-2">
                    Henüz Bölüm Yok
                  </h3>
                  <p className="text-primary-foreground/70 mb-6">
                    Bu anime için henüz bölüm eklenmemiş.
                  </p>
                  {isAdmin && (
                    <Button 
                      onClick={openNewEpisodeDialog}
                      className="bg-gradient-to-r from-accentneongreen to-accentneongreen/80 text-primary hover:from-accentneongreen/90 hover:to-accentneongreen/70"
                    >
                      <Plus className="w-4 h-4 mr-2" />
                      İlk Bölümü Ekle
                    </Button>
                  )}
                </motion.div>
              )}
            </TabsContent>

            {/* Watch Tab */}
            <TabsContent value="watch" className="space-y-6">
              {selectedEpisode ? (
                <motion.div
                  initial={{ y: 20, opacity: 0 }}
                  animate={{ y: 0, opacity: 1 }}
                  className="space-y-6"
                >
                  {/* Video Player */}
                  <div className="bg-gradient-to-br from-secondary/20 to-primary/30 backdrop-blur-sm rounded-2xl p-6 border border-accentneongreen/20">
                    <div className="mb-4">
                      <h2 className="font-heading text-2xl font-bold text-primary-foreground mb-2">
                        Bölüm {selectedEpisode.episodeNumber}: {selectedEpisode.episodeTitle}
                      </h2>
                      <p className="text-primary-foreground/70">
                        {selectedEpisode.episodeDescription}
                      </p>
                    </div>

                    {selectedEpisode.videoUrl ? (
                      <div className="relative bg-black rounded-xl overflow-hidden">
                        {isYouTubeUrl(selectedEpisode.videoUrl) ? (
                          // YouTube Embed Player
                          <div className="aspect-video">
                            <iframe
                              src={getYouTubeEmbedUrl(selectedEpisode.videoUrl)}
                              title={`${selectedEpisode.episodeTitle} - YouTube Video`}
                              className="w-full h-full rounded-xl"
                              allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                              allowFullScreen
                            />
                          </div>
                        ) : (
                          // Enhanced Video Player for uploaded files
                          <EnhancedVideoPlayer
                            src={selectedEpisode.videoUrl}
                            poster={selectedEpisode.thumbnail}
                            title={selectedEpisode.episodeTitle}
                            onTimeUpdate={(currentTime, duration) => {
                              setCurrentTime(currentTime);
                              setVideoDuration(duration);
                              setVideoProgress((currentTime / duration) * 100);
                            }}
                            onPlay={() => setIsVideoPlaying(true)}
                            onPause={() => setIsVideoPlaying(false)}
                            className="aspect-video"
                          />
                        )}
                        
                        {/* Video Info Overlay - Only for non-YouTube videos */}
                        {!isYouTubeUrl(selectedEpisode.videoUrl) && (
                          <div className="absolute bottom-4 left-4 right-4">
                            <div className="bg-black/50 backdrop-blur-sm rounded-lg p-3">
                              <div className="flex items-center justify-between text-white text-sm">
                                <span>{formatTime(currentTime)} / {formatTime(videoDuration)}</span>
                                <div className="flex items-center space-x-2">
                                  {selectedEpisode.duration && (
                                    <Badge className="bg-accentneongreen/20 text-accentneongreen border-accentneongreen/30">
                                      {selectedEpisode.duration} dk
                                    </Badge>
                                  )}
                                </div>
                              </div>
                            </div>
                          </div>
                        )}
                        
                        {/* YouTube Badge */}
                        {isYouTubeUrl(selectedEpisode.videoUrl) && (
                          <div className="absolute top-4 right-4">
                            <Badge className="bg-red-600 text-white border-0 shadow-lg flex items-center space-x-1">
                              <Youtube className="w-3 h-3" />
                              <span>YouTube</span>
                            </Badge>
                          </div>
                        )}
                      </div>
                    ) : (
                      <div className="aspect-video bg-gradient-to-br from-secondary/30 to-primary/20 rounded-xl flex items-center justify-center">
                        <div className="text-center">
                          <Video className="w-16 h-16 text-primary-foreground/40 mx-auto mb-4" />
                          <p className="text-primary-foreground/60">Bu bölüm için video mevcut değil</p>
                        </div>
                      </div>
                    )}
                  </div>

                  {/* Episode Navigation */}
                  <div className="bg-gradient-to-br from-secondary/20 to-primary/30 backdrop-blur-sm rounded-2xl p-6 border border-accentneongreen/20">
                    <h3 className="font-heading text-xl font-bold text-primary-foreground mb-4">
                      Diğer Bölümler
                    </h3>
                    
                    <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-6 gap-3">
                      {episodes.map((episode) => (
                        <motion.div
                          key={episode._id}
                          whileHover={{ scale: 1.05 }}
                          whileTap={{ scale: 0.95 }}
                        >
                          <Button
                            variant={selectedEpisode._id === episode._id ? "default" : "outline"}
                            size="sm"
                            className={`w-full ${
                              selectedEpisode._id === episode._id
                                ? 'bg-accentneongreen text-primary'
                                : 'border-primary-foreground/30 text-primary-foreground hover:bg-primary-foreground/10'
                            }`}
                            onClick={() => setSelectedEpisode(episode)}
                          >
                            {episode.episodeNumber}
                          </Button>
                        </motion.div>
                      ))}
                    </div>
                  </div>
                </motion.div>
              ) : (
                <motion.div
                  initial={{ opacity: 0 }}
                  animate={{ opacity: 1 }}
                  className="text-center py-16"
                >
                  <div className="w-24 h-24 bg-accentneongreen/20 rounded-full flex items-center justify-center mx-auto mb-6">
                    <Video className="w-12 h-12 text-accentneongreen" />
                  </div>
                  <h3 className="font-heading text-2xl font-bold text-primary-foreground mb-2">
                    Bölüm Seçin
                  </h3>
                  <p className="text-primary-foreground/70 mb-6">
                    İzlemek için bir bölüm seçin veya bölümler sekmesinden başlayın.
                  </p>
                  <Button 
                    onClick={() => setActiveTab('episodes')}
                    className="bg-gradient-to-r from-accentneongreen to-accentneongreen/80 text-primary hover:from-accentneongreen/90 hover:to-accentneongreen/70"
                  >
                    <Play className="w-4 h-4 mr-2" />
                    Bölümleri Görüntüle
                  </Button>
                </motion.div>
              )}
            </TabsContent>
          </Tabs>
        </motion.div>

        {/* Episode Dialog */}
        <AnimatePresence>
          {isEpisodeDialogOpen && (
            <Dialog open={isEpisodeDialogOpen} onOpenChange={setIsEpisodeDialogOpen}>
              <DialogContent className="bg-gradient-to-br from-secondary/90 to-primary/80 backdrop-blur-xl border border-accentneongreen/30 max-w-3xl max-h-[90vh] overflow-y-auto">
                <motion.div
                  initial={{ scale: 0.9, opacity: 0 }}
                  animate={{ scale: 1, opacity: 1 }}
                  exit={{ scale: 0.9, opacity: 0 }}
                  transition={{ duration: 0.2 }}
                >
                  <DialogHeader className="pb-6">
                    <DialogTitle className="text-2xl font-bold text-primary-foreground flex items-center">
                      <div className="p-2 bg-accentneongreen/20 rounded-lg mr-3">
                        <Play className="w-6 h-6 text-accentneongreen" />
                      </div>
                      {editingEpisode ? 'Bölüm Düzenle' : 'Yeni Bölüm Ekle'}
                    </DialogTitle>
                  </DialogHeader>
                  
                  <div className="space-y-6">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div className="space-y-2">
                        <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                          <Film className="w-4 h-4 mr-2 text-accentneongreen" />
                          Bölüm Numarası
                        </label>
                        <Input
                          type="number"
                          value={editingEpisode ? editingEpisode.episodeNumber || '' : newEpisode.episodeNumber}
                          onChange={(e) => editingEpisode 
                            ? setEditingEpisode({...editingEpisode, episodeNumber: parseInt(e.target.value)})
                            : setNewEpisode({...newEpisode, episodeNumber: parseInt(e.target.value)})
                          }
                          className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                          placeholder="1"
                        />
                      </div>
                      <div className="space-y-2">
                        <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                          <BookOpen className="w-4 h-4 mr-2 text-accentneongreen" />
                          Bölüm Başlığı
                        </label>
                        <Input
                          value={editingEpisode ? editingEpisode.episodeTitle || '' : newEpisode.episodeTitle}
                          onChange={(e) => editingEpisode 
                            ? setEditingEpisode({...editingEpisode, episodeTitle: e.target.value})
                            : setNewEpisode({...newEpisode, episodeTitle: e.target.value})
                          }
                          className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                          placeholder="Bölüm başlığını girin..."
                        />
                      </div>
                    </div>
                    
                    <div className="space-y-2">
                      <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                        <BookOpen className="w-4 h-4 mr-2 text-accentneongreen" />
                        Bölüm Açıklaması
                      </label>
                      <Textarea
                        value={editingEpisode ? editingEpisode.episodeDescription || '' : newEpisode.episodeDescription}
                        onChange={(e) => editingEpisode 
                          ? setEditingEpisode({...editingEpisode, episodeDescription: e.target.value})
                          : setNewEpisode({...newEpisode, episodeDescription: e.target.value})
                        }
                        className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300 min-h-[100px]"
                        placeholder="Bölüm açıklamasını girin..."
                        rows={4}
                      />
                    </div>
                    
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div className="space-y-2">
                        <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                          <Video className="w-4 h-4 mr-2 text-accentneongreen" />
                          Video URL (YouTube veya Doğrudan Link)
                        </label>
                        <Input
                          value={editingEpisode ? editingEpisode.videoUrl || '' : newEpisode.videoUrl}
                          onChange={(e) => editingEpisode 
                            ? setEditingEpisode({...editingEpisode, videoUrl: e.target.value})
                            : setNewEpisode({...newEpisode, videoUrl: e.target.value})
                          }
                          className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                          placeholder="https://www.youtube.com/watch?v=... veya https://example.com/video.mp4"
                        />
                        <div className="text-xs text-primary-foreground/60 mt-1">
                          YouTube linklerini veya doğrudan video dosya linklerini destekler
                        </div>
                      </div>
                      <div className="space-y-2">
                        <label className="text-sm font-medium text-primary-foreground/80 flex items-center">
                          <Clock className="w-4 h-4 mr-2 text-accentneongreen" />
                          Süre (dakika)
                        </label>
                        <Input
                          type="number"
                          value={editingEpisode ? editingEpisode.duration || '' : newEpisode.duration}
                          onChange={(e) => editingEpisode 
                            ? setEditingEpisode({...editingEpisode, duration: parseInt(e.target.value)})
                            : setNewEpisode({...newEpisode, duration: parseInt(e.target.value)})
                          }
                          className="bg-primary/50 border-accentneongreen/30 text-primary-foreground focus:border-accentneongreen transition-all duration-300"
                          placeholder="24"
                        />
                      </div>
                    </div>
                    
                    <div className="flex space-x-3 pt-4">
                      <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }} className="flex-1">
                        <Button 
                          onClick={editingEpisode ? handleEditEpisode : handleAddEpisode} 
                          className="w-full bg-gradient-to-r from-accentneongreen to-accentneongreen/80 text-primary hover:from-accentneongreen/90 hover:to-accentneongreen/70 shadow-lg transition-all duration-300"
                        >
                          <Save className="w-4 h-4 mr-2" />
                          {editingEpisode ? 'Güncelle' : 'Ekle'}
                        </Button>
                      </motion.div>
                      <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                        <Button 
                          onClick={() => setIsEpisodeDialogOpen(false)} 
                          variant="outline"
                          className="border-primary-foreground/30 text-primary-foreground hover:bg-primary-foreground/10 transition-all duration-300"
                        >
                          <X className="w-4 h-4 mr-2" />
                          İptal
                        </Button>
                      </motion.div>
                    </div>
                  </div>
                </motion.div>
              </DialogContent>
            </Dialog>
          )}
        </AnimatePresence>
      </div>

      {/* Related Anime Section */}
      {relatedAnime.length > 0 && (
        <section className="w-full max-w-[120rem] mx-auto px-6 pb-16">
          <motion.div
            initial={{ y: 30, opacity: 0 }}
            animate={{ y: 0, opacity: 1 }}
            transition={{ delay: 0.5 }}
          >
            <h2 className="font-heading text-3xl font-bold text-primary-foreground mb-8">
              <span className="text-accentneongreen">{anime.genre}</span> Türünde Daha Fazla
            </h2>
            
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
              <AnimatePresence>
                {relatedAnime.map((relatedItem, index) => (
                  <motion.div
                    key={relatedItem._id}
                    initial={{ y: 20, opacity: 0 }}
                    animate={{ y: 0, opacity: 1 }}
                    transition={{ delay: index * 0.1 }}
                    whileHover={{ y: -5, scale: 1.02 }}
                  >
                    <Link to={`/anime/${relatedItem._id}`}>
                      <Card className="bg-gradient-to-br from-secondary/20 to-primary/30 border border-accentneongreen/20 backdrop-blur-sm hover:border-accentneongreen/40 transition-all duration-300 group overflow-hidden">
                        <div className="aspect-[3/4] relative overflow-hidden">
                          {(relatedItem.posterImageUrl || relatedItem.posterImage) ? (
                            <Image
                              src={relatedItem.posterImageUrl || relatedItem.posterImage || ''}
                              alt={relatedItem.title || 'Anime posteri'}
                              className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                              width={250}
                            />
                          ) : (
                            <div className="w-full h-full bg-gradient-to-br from-secondary/30 to-primary/20 flex items-center justify-center">
                              <Play className="w-8 h-8 text-primary-foreground/40" />
                            </div>
                          )}
                        </div>
                        <CardContent className="p-4">
                          <h3 className="font-heading text-sm font-bold text-primary-foreground line-clamp-2 group-hover:text-accentneongreen transition-colors">
                            {relatedItem.title}
                          </h3>
                          {relatedItem.releaseYear && (
                            <p className="font-paragraph text-primary-foreground/60 text-xs mt-1">
                              {relatedItem.releaseYear}
                            </p>
                          )}
                        </CardContent>
                      </Card>
                    </Link>
                  </motion.div>
                ))}
              </AnimatePresence>
            </div>
          </motion.div>
        </section>
      )}

      {/* Footer */}
      <footer className="w-full bg-primary border-t border-primary-foreground/10">
        <div className="max-w-[120rem] mx-auto px-6 py-12">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="space-y-4">
              <div className="flex items-center space-x-2">
                <div className="w-8 h-8 bg-accentneongreen rounded-lg flex items-center justify-center">
                  <BookOpen className="w-5 h-5 text-primary" />
                </div>
                <span className="font-heading text-xl font-bold text-primary-foreground">Mangius</span>
              </div>
              <p className="font-paragraph text-primary-foreground/70">
                En iyi anime ve manga içeriklerine açılan kapınız.
              </p>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Gözat</h3>
              <div className="space-y-2">
                <Link to="/anime" className="block font-paragraph text-accentneongreen">
                  Anime Koleksiyonu
                </Link>
                <Link to="/manga" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Manga Kütüphanesi
                </Link>
                <Link to="/" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Ana Sayfa
                </Link>
              </div>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Destek</h3>
              <div className="space-y-2">
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Yardım Merkezi
                </a>
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  İletişim
                </a>
              </div>
            </div>
          </div>
          
          <div className="border-t border-primary-foreground/10 mt-8 pt-8 text-center">
            <p className="font-paragraph text-primary-foreground/60">
              © 2024 Mangius. Tüm hakları saklıdır.
            </p>
          </div>
        </div>
      </footer>

      {/* Episode Upload Dialog */}
      <AnimeEpisodeUploadDialog
        isOpen={isEpisodeUploadDialogOpen}
        onClose={() => setIsEpisodeUploadDialogOpen(false)}
        animeId={id || ''}
        animeTitle={anime?.title}
        onEpisodeAdded={handleEpisodeAdded}
      />
    </div>
  );
}