import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { BaseCrudService } from '@/integrations';
import { useMember } from '@/integrations';
import { AnimeTitles } from '@/entities';
import { useFavorites } from '@/hooks/useFavorites';
import { useAdminAccess } from '@/hooks/useAdminAccess';
import { Image } from '@/components/ui/image';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { BookOpen, Search, Filter, Calendar, Play, User, LogOut, Star, Heart, Plus, Settings, Upload } from 'lucide-react';
import { VideoUploadDialog } from '@/components/ui/video-upload-dialog';

export default function AnimePage() {
  const [animeList, setAnimeList] = useState<AnimeTitles[]>([]);
  const [filteredAnime, setFilteredAnime] = useState<AnimeTitles[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedGenre, setSelectedGenre] = useState<string>('all');
  const [sortBy, setSortBy] = useState<string>('title');
  const [isUploadDialogOpen, setIsUploadDialogOpen] = useState(false);
  const { member, isAuthenticated, actions } = useMember();
  
  // Admin access check
  const { isAdmin } = useAdminAccess();
  
  // Favorites hook
  const { addToFavorites, removeFromFavorites, isFavorite } = useFavorites(
    member?.loginEmail
  );

  useEffect(() => {
    const fetchAnime = async () => {
      try {
        const response = await BaseCrudService.getAll<AnimeTitles>('animetitles');
        setAnimeList(response.items);
        setFilteredAnime(response.items);
      } catch (error) {
        console.error('Error fetching anime:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchAnime();
  }, []);

  useEffect(() => {
    let filtered = [...animeList];

    // Filter by search term
    if (searchTerm) {
      filtered = filtered.filter(anime =>
        anime.title?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        anime.synopsis?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        anime.creator?.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    // Filter by genre
    if (selectedGenre !== 'all') {
      filtered = filtered.filter(anime => anime.genre === selectedGenre);
    }

    // Sort
    filtered.sort((a, b) => {
      switch (sortBy) {
        case 'title':
          return (a.title || '').localeCompare(b.title || '');
        case 'year':
          return (b.releaseYear || 0) - (a.releaseYear || 0);
        case 'episodes':
          return (b.numberOfEpisodes || 0) - (a.numberOfEpisodes || 0);
        default:
          return 0;
      }
    });

    setFilteredAnime(filtered);
  }, [animeList, searchTerm, selectedGenre, sortBy]);

  const genres = Array.from(new Set(animeList.map(anime => anime.genre).filter(Boolean)));

  const handleFavoriteToggle = async (animeId: string, e: React.MouseEvent) => {
    e.preventDefault(); // Prevent navigation when clicking favorite button
    e.stopPropagation();
    
    if (!isAuthenticated) {
      actions.login();
      return;
    }

    const isCurrentlyFavorite = isFavorite(animeId, 'anime');
    
    if (isCurrentlyFavorite) {
      await removeFromFavorites(animeId, 'anime');
    } else {
      await addToFavorites(animeId, 'anime');
    }
  };

  const handleAnimeAdded = () => {
    // Refresh anime list after new anime is added
    const fetchAnime = async () => {
      try {
        const response = await BaseCrudService.getAll<AnimeTitles>('animetitles');
        setAnimeList(response.items);
        setFilteredAnime(response.items);
      } catch (error) {
        console.error('Error fetching anime:', error);
      }
    };
    fetchAnime();
  };

  if (isLoading) {
    return (
      <div className="min-h-screen bg-primary flex items-center justify-center">
        <div className="text-primary-foreground text-xl">Anime koleksiyonu yükleniyor...</div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-primary">
      {/* Navigation */}
      <nav className="w-full px-6 py-4 flex items-center justify-between max-w-[120rem] mx-auto border-b border-primary-foreground/10">
        <Link to="/" className="flex items-center space-x-2">
          <div className="w-8 h-8 bg-accentneongreen rounded-lg flex items-center justify-center">
            <BookOpen className="w-5 h-5 text-primary" />
          </div>
          <span className="font-heading text-xl font-bold text-primary-foreground">Mangius</span>
        </Link>
        
        <div className="hidden md:flex items-center space-x-8">
          <Link to="/anime" className="font-paragraph text-accentneongreen font-medium">
            Anime
          </Link>
          <Link to="/manga" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Manga
          </Link>
          {isAuthenticated && (
            <Link to="/friends" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
              Arkadaşlar
            </Link>
          )}
          <Link to="/" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Ana Sayfa
          </Link>
        </div>

        <div className="flex items-center space-x-4">
          {!isAuthenticated ? (
            <Button 
              onClick={actions.login}
              variant="outline" 
              className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary"
            >
              Giriş Yap
            </Button>
          ) : (
            <>
              {isAdmin && (
                <Link to="/admin">
                  <Button variant="outline" className="border-secondary text-secondary hover:bg-secondary hover:text-secondary-foreground">
                    <Settings className="w-4 h-4 mr-2" />
                    Admin
                  </Button>
                </Link>
              )}
              <Link to="/profile">
                <Button variant="outline" className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary">
                  <User className="w-4 h-4 mr-2" />
                  {member?.profile?.nickname || 'Profil'}
                </Button>
              </Link>
              <Button 
                onClick={actions.logout}
                variant="outline" 
                className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary"
              >
                <LogOut className="w-4 h-4 mr-2" />
                Çıkış Yap
              </Button>
            </>
          )}
        </div>
      </nav>

      {/* Header Section */}
      <section className="w-full max-w-[120rem] mx-auto px-6 py-16">
        <div className="text-center mb-12">
          <h1 className="font-heading text-5xl md:text-6xl font-black text-primary-foreground mb-4">
            <span className="text-accentneongreen">Anime</span> Koleksiyonu
          </h1>
          <p className="font-paragraph text-xl text-primary-foreground/70 max-w-2xl mx-auto">
            Çeşitli türlerden ve yaratıcılardan harika anime serilerini keşfedin
          </p>
        </div>

        {/* Filters and Search */}
        <div className="bg-secondary rounded-2xl p-6 mb-12">
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-secondary-foreground/60 w-4 h-4" />
              <Input
                placeholder="Anime ara..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 bg-primary border-primary-foreground/20 text-primary-foreground placeholder:text-primary-foreground/60"
              />
            </div>
            
            <Select value={selectedGenre} onValueChange={setSelectedGenre}>
              <SelectTrigger className="bg-primary border-primary-foreground/20 text-primary-foreground">
                <Filter className="w-4 h-4 mr-2" />
                <SelectValue placeholder="Tür" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Tüm Türler</SelectItem>
                {genres.map((genre) => (
                  <SelectItem key={genre} value={genre || ''}>
                    {genre}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            <Select value={sortBy} onValueChange={setSortBy}>
              <SelectTrigger className="bg-primary border-primary-foreground/20 text-primary-foreground">
                <SelectValue placeholder="Sırala" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="title">İsim A-Z</SelectItem>
                <SelectItem value="year">Çıkış Yılı</SelectItem>
                <SelectItem value="episodes">Bölüm Sayısı</SelectItem>
              </SelectContent>
            </Select>

            <div className="flex items-center justify-between">
              <span className="font-paragraph text-secondary-foreground/70 text-sm">
                {filteredAnime.length} anime bulundu
              </span>
              <div className="flex items-center gap-3">
                <Badge className="bg-accentneongreen text-primary">
                  {animeList.filter(anime => anime.isFeatured).length} Öne Çıkan
                </Badge>
                {isAdmin && (
                  <Button
                    onClick={() => setIsUploadDialogOpen(true)}
                    size="sm"
                    className="bg-secondary text-secondary-foreground hover:bg-secondary/80"
                  >
                    <Upload className="w-4 h-4 mr-2" />
                    Anime Ekle
                  </Button>
                )}
              </div>
            </div>
          </div>
        </div>

        {/* Anime Grid */}
        {filteredAnime.length === 0 ? (
          <div className="text-center py-16">
            <div className="w-16 h-16 bg-secondary rounded-2xl mx-auto mb-4 flex items-center justify-center">
              <Search className="w-8 h-8 text-secondary-foreground/60" />
            </div>
            <h3 className="font-heading text-2xl font-bold text-primary-foreground mb-2">
              Anime bulunamadı
            </h3>
            <p className="font-paragraph text-primary-foreground/70">
              Arama veya filtre kriterlerinizi ayarlamayı deneyin
            </p>
          </div>
        ) : (
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 2xl:grid-cols-5 gap-6">
            {filteredAnime.map((anime) => (
              <Card key={anime._id} className="group bg-secondary border-0 overflow-hidden hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-1">
                <div className="relative">
                  {(anime.posterImageUrl || anime.posterImage) ? (
                    <Image
                      src={anime.posterImageUrl || anime.posterImage || ''}
                      alt={anime.title || 'Anime kapak görseli'}
                      className="w-full h-80 object-cover"
                      width={300}
                    />
                  ) : (
                    <div className="w-full h-80 bg-secondary-foreground/10 flex items-center justify-center">
                      <Play className="w-16 h-16 text-secondary-foreground/30" />
                    </div>
                  )}
                  
                  {/* Overlay */}
                  <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/20 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300" />
                  
                  {/* Featured Badge */}
                  {anime.isFeatured && (
                    <Badge className="absolute top-3 left-3 bg-accentneongreen text-primary font-medium px-2 py-1">
                      ⭐ Öne Çıkan
                    </Badge>
                  )}
                  
                  {/* Genre Badge */}
                  {anime.genre && (
                    <Badge className="absolute top-3 right-3 bg-primary/80 text-primary-foreground font-medium px-2 py-1">
                      {anime.genre}
                    </Badge>
                  )}
                  
                  {/* Quick Action Buttons */}
                  <div className="absolute bottom-4 left-4 right-4 opacity-0 group-hover:opacity-100 transition-all duration-300 transform translate-y-4 group-hover:translate-y-0">
                    <div className="flex gap-2">
                      <Link to={`/anime/${anime._id}`} className="flex-1">
                        <Button size="sm" className="w-full bg-accentneongreen text-primary hover:bg-accentneongreen/90 font-medium">
                          <Play className="w-4 h-4 mr-2" />
                          İzle
                        </Button>
                      </Link>
                      <Button size="sm" variant="outline" className="border-white/50 text-white hover:bg-white/20 backdrop-blur-sm">
                        <Heart 
                          className={`w-4 h-4 ${isFavorite(anime._id, 'anime') ? 'fill-red-500 text-red-500' : ''}`}
                          onClick={(e) => handleFavoriteToggle(anime._id, e)}
                        />
                      </Button>
                    </div>
                  </div>
                </div>
                
                <CardContent className="p-4 space-y-3">
                  <div>
                    <h3 className="font-heading text-lg font-bold text-secondary-foreground mb-1 line-clamp-1 group-hover:text-accentneongreen transition-colors">
                      {anime.title || 'Başlık Yok'}
                    </h3>
                    <p className="font-paragraph text-secondary-foreground/70 text-sm line-clamp-2 leading-relaxed">
                      {anime.synopsis || 'Açıklama mevcut değil.'}
                    </p>
                  </div>
                  
                  {/* Info Row */}
                  <div className="flex items-center justify-between pt-2 border-t border-secondary-foreground/10">
                    <div className="flex items-center space-x-1 text-xs text-secondary-foreground/60">
                      <Calendar className="w-3 h-3" />
                      <span>{anime.releaseYear || 'Bilinmiyor'}</span>
                    </div>
                    <div className="flex items-center space-x-1 text-xs text-secondary-foreground/60">
                      <Play className="w-3 h-3" />
                      <span>{anime.numberOfEpisodes || '?'} Bölüm</span>
                    </div>
                  </div>
                  
                  {/* Creator */}
                  {anime.creator && (
                    <div className="flex items-center space-x-1 text-xs text-secondary-foreground/50">
                      <User className="w-3 h-3" />
                      <span>{anime.creator}</span>
                    </div>
                  )}
                  
                  {/* Action Buttons */}
                  <div className="flex gap-2 pt-2">
                    <Link to={`/anime/${anime._id}`} className="flex-1">
                      <Button size="sm" variant="outline" className="w-full border-secondary-foreground/20 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary text-xs">
                        Detaylar
                      </Button>
                    </Link>
                    <Button size="sm" variant="outline" className="border-accentneongreen/30 text-accentneongreen hover:bg-accentneongreen hover:text-primary text-xs">
                      <Heart 
                        className={`w-3 h-3 mr-1 ${isFavorite(anime._id, 'anime') ? 'fill-red-500 text-red-500' : ''}`}
                        onClick={(e) => handleFavoriteToggle(anime._id, e)}
                      />
                      {isFavorite(anime._id, 'anime') ? 'Favoride' : 'Favori'}
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}
      </section>

      {/* Footer */}
      <footer className="w-full bg-primary border-t border-primary-foreground/10 mt-16">
        <div className="max-w-[120rem] mx-auto px-6 py-12">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="space-y-4">
              <div className="flex items-center space-x-2">
                <div className="w-8 h-8 bg-accentneongreen rounded-lg flex items-center justify-center">
                  <BookOpen className="w-5 h-5 text-primary" />
                </div>
                <span className="font-heading text-xl font-bold text-primary-foreground">Mangius</span>
              </div>
              <p className="font-paragraph text-primary-foreground/70">
                En iyi anime ve manga içeriklerine açılan kapınız.
              </p>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Gözat</h3>
              <div className="space-y-2">
                <Link to="/anime" className="block font-paragraph text-accentneongreen">
                  Anime Koleksiyonu
                </Link>
                <Link to="/manga" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Manga Kütüphanesi
                </Link>
                <Link to="/" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Ana Sayfa
                </Link>
              </div>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Destek</h3>
              <div className="space-y-2">
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Yardım Merkezi
                </a>
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  İletişim
                </a>
              </div>
            </div>
          </div>
          
          <div className="border-t border-primary-foreground/10 mt-8 pt-8 text-center">
            <p className="font-paragraph text-primary-foreground/60">
              © 2024 Mangius. Tüm hakları saklıdır.
            </p>
          </div>
        </div>
      </footer>

      {/* Upload Dialog */}
      <VideoUploadDialog
        isOpen={isUploadDialogOpen}
        onClose={() => setIsUploadDialogOpen(false)}
        onAnimeAdded={handleAnimeAdded}
      />
    </div>
  );
}