import React, { useState, useEffect } from 'react';
import { useMember } from '@/integrations';
import { useFriendship } from '@/hooks/useFriendship';
import { BaseCrudService } from '@/integrations';
import { KullancProfilleri } from '@/entities';
import type { Member } from '@/types/member';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Users, UserPlus, MessageCircle, Check, X, Search } from 'lucide-react';
import { Link } from 'react-router-dom';

export default function FriendsPage() {
  const { member }: { member: Member | null } = useMember();
  const {
    friendRequests,
    friends,
    loading,
    sendFriendRequest,
    acceptFriendRequest,
    rejectFriendRequest,
    areFriends
  } = useFriendship();

  const [allUsers, setAllUsers] = useState<KullancProfilleri[]>([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedUser, setSelectedUser] = useState<KullancProfilleri | null>(null);
  const [requestMessage, setRequestMessage] = useState('');

  useEffect(() => {
    loadAllUsers();
  }, []);

  const loadAllUsers = async () => {
    try {
      const { items } = await BaseCrudService.getAll<KullancProfilleri>('kullancprofilleri');
      // Kendi profilini hariç tut
      const otherUsers = items.filter(user => user.memberId !== member?._id);
      setAllUsers(otherUsers);
    } catch (error) {
      console.error('Kullanıcılar yüklenemedi:', error);
    }
  };

  const filteredUsers = allUsers.filter(user =>
    user.displayName?.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handleSendRequest = async () => {
    if (selectedUser?.memberId) {
      await sendFriendRequest(selectedUser.memberId, requestMessage);
      setSelectedUser(null);
      setRequestMessage('');
    }
  };

  const getFriendProfile = async (userId: string) => {
    try {
      const { items } = await BaseCrudService.getAll<KullancProfilleri>('kullancprofilleri');
      return items.find(user => user.memberId === userId);
    } catch (error) {
      console.error('Profil yüklenemedi:', error);
      return null;
    }
  };

  const [friendProfiles, setFriendProfiles] = useState<{ [key: string]: KullancProfilleri }>({});

  useEffect(() => {
    const loadFriendProfiles = async () => {
      const profiles: { [key: string]: KullancProfilleri } = {};
      
      for (const friendship of friends) {
        const friendId = friendship.userOneId === member?._id 
          ? friendship.userTwoId 
          : friendship.userOneId;
        
        if (friendId) {
          const profile = await getFriendProfile(friendId);
          if (profile) {
            profiles[friendId] = profile;
          }
        }
      }
      
      setFriendProfiles(profiles);
    };

    if (friends.length > 0) {
      loadFriendProfiles();
    }
  }, [friends, member?._id]);

  return (
    <div className="min-h-screen bg-black text-white">
      {/* Header */}
      <header className="bg-black border-b border-gray-800 sticky top-0 z-50">
        <div className="max-w-[120rem] mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <Link to="/" className="text-2xl font-bold text-accentneongreen">
              AnimeHub
            </Link>
            <nav className="flex items-center space-x-8">
              <Link to="/" className="hover:text-accentneongreen transition-colors">Ana Sayfa</Link>
              <Link to="/anime" className="hover:text-accentneongreen transition-colors">Anime</Link>
              <Link to="/manga" className="hover:text-accentneongreen transition-colors">Manga</Link>
              <Link to="/friends" className="text-accentneongreen">Arkadaşlar</Link>
              <Link to="/profile" className="hover:text-accentneongreen transition-colors">Profil</Link>
            </nav>
          </div>
        </div>
      </header>

      <div className="max-w-[120rem] mx-auto px-6 py-8">
        <div className="mb-8">
          <h1 className="text-4xl font-bold mb-2">Arkadaşlarım</h1>
          <p className="text-gray-400">Arkadaşlarınızı yönetin ve yeni arkadaşlar edinin</p>
        </div>

        <Tabs defaultValue="friends" className="w-full">
          <TabsList className="grid w-full grid-cols-3 bg-gray-900">
            <TabsTrigger value="friends" className="data-[state=active]:bg-accentneongreen data-[state=active]:text-black">
              <Users className="w-4 h-4 mr-2" />
              Arkadaşlarım ({friends.length})
            </TabsTrigger>
            <TabsTrigger value="requests" className="data-[state=active]:bg-accentneongreen data-[state=active]:text-black">
              <UserPlus className="w-4 h-4 mr-2" />
              İstekler ({friendRequests.length})
            </TabsTrigger>
            <TabsTrigger value="discover" className="data-[state=active]:bg-accentneongreen data-[state=active]:text-black">
              <Search className="w-4 h-4 mr-2" />
              Keşfet
            </TabsTrigger>
          </TabsList>

          <TabsContent value="friends" className="mt-6">
            {friends.length === 0 ? (
              <Card className="bg-gray-900 border-gray-800">
                <CardContent className="p-8 text-center">
                  <Users className="w-16 h-16 mx-auto mb-4 text-gray-600" />
                  <h3 className="text-xl font-semibold mb-2">Henüz arkadaşınız yok</h3>
                  <p className="text-gray-400 mb-4">Yeni arkadaşlar bulmak için "Keşfet" sekmesini kullanın</p>
                </CardContent>
              </Card>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {friends.map((friendship) => {
                  const friendId = friendship.userOneId === member?._id 
                    ? friendship.userTwoId 
                    : friendship.userOneId;
                  const friendProfile = friendProfiles[friendId || ''];
                  
                  return (
                    <Card key={friendship._id} className="bg-gray-900 border-gray-800 hover:border-accentneongreen transition-colors">
                      <CardContent className="p-6">
                        <div className="flex items-center space-x-4 mb-4">
                          <Avatar className="w-12 h-12">
                            <AvatarImage src={friendProfile?.profilePicture} />
                            <AvatarFallback className="bg-accentneongreen text-black">
                              {friendProfile?.displayName?.charAt(0) || 'U'}
                            </AvatarFallback>
                          </Avatar>
                          <div>
                            <h3 className="font-semibold">{friendProfile?.displayName || 'Bilinmeyen Kullanıcı'}</h3>
                            <p className="text-sm text-gray-400">
                              {new Date(friendship.friendshipDate || '').toLocaleDateString('tr-TR')} tarihinden beri arkadaş
                            </p>
                          </div>
                        </div>
                        <Link to={`/messages/${friendId}`}>
                          <Button className="w-full bg-accentneongreen text-black hover:bg-accentneongreen/80">
                            <MessageCircle className="w-4 h-4 mr-2" />
                            Mesaj Gönder
                          </Button>
                        </Link>
                      </CardContent>
                    </Card>
                  );
                })}
              </div>
            )}
          </TabsContent>

          <TabsContent value="requests" className="mt-6">
            {friendRequests.length === 0 ? (
              <Card className="bg-gray-900 border-gray-800">
                <CardContent className="p-8 text-center">
                  <UserPlus className="w-16 h-16 mx-auto mb-4 text-gray-600" />
                  <h3 className="text-xl font-semibold mb-2">Bekleyen arkadaşlık isteği yok</h3>
                  <p className="text-gray-400">Yeni arkadaşlık istekleri burada görünecek</p>
                </CardContent>
              </Card>
            ) : (
              <div className="space-y-4">
                {friendRequests.map((request) => (
                  <Card key={request._id} className="bg-gray-900 border-gray-800">
                    <CardContent className="p-6">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-4">
                          <Avatar className="w-12 h-12">
                            <AvatarFallback className="bg-accentneongreen text-black">
                              U
                            </AvatarFallback>
                          </Avatar>
                          <div>
                            <h3 className="font-semibold">Arkadaşlık İsteği</h3>
                            <p className="text-sm text-gray-400">
                              {new Date(request.requestDate || '').toLocaleDateString('tr-TR')}
                            </p>
                            {request.message && (
                              <p className="text-sm text-gray-300 mt-1">"{request.message}"</p>
                            )}
                          </div>
                        </div>
                        <div className="flex space-x-2">
                          <Button
                            onClick={() => acceptFriendRequest(request._id, request.senderUserId || '')}
                            disabled={loading}
                            className="bg-green-600 hover:bg-green-700"
                          >
                            <Check className="w-4 h-4 mr-2" />
                            Kabul Et
                          </Button>
                          <Button
                            onClick={() => rejectFriendRequest(request._id)}
                            disabled={loading}
                            variant="destructive"
                          >
                            <X className="w-4 h-4 mr-2" />
                            Reddet
                          </Button>
                          <Link to={`/messages/${request.senderUserId}`}>
                            <Button
                              variant="outline"
                              className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-black"
                            >
                              <MessageCircle className="w-4 h-4 mr-2" />
                              DM
                            </Button>
                          </Link>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>

          <TabsContent value="discover" className="mt-6">
            <div className="mb-6">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                <Input
                  placeholder="Kullanıcı ara..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10 bg-gray-900 border-gray-700 text-white"
                />
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {filteredUsers.map((user) => (
                <Card key={user._id} className="bg-gray-900 border-gray-800 hover:border-accentneongreen transition-colors">
                  <CardContent className="p-6">
                    <div className="flex items-center space-x-4 mb-4">
                      <Avatar className="w-12 h-12">
                        <AvatarImage src={user.profilePicture} />
                        <AvatarFallback className="bg-accentneongreen text-black">
                          {user.displayName?.charAt(0) || 'U'}
                        </AvatarFallback>
                      </Avatar>
                      <div>
                        <h3 className="font-semibold">{user.displayName || 'Bilinmeyen Kullanıcı'}</h3>
                        <p className="text-sm text-gray-400">
                          {user.aboutMe ? user.aboutMe.substring(0, 50) + '...' : 'Açıklama yok'}
                        </p>
                      </div>
                    </div>
                    
                    {areFriends(user.memberId || '') ? (
                      <div className="space-y-2">
                        <Badge className="w-full justify-center bg-green-600">
                          <Check className="w-4 h-4 mr-2" />
                          Arkadaş
                        </Badge>
                        <Link to={`/messages/${user.memberId}`} className="w-full">
                          <Button 
                            variant="outline" 
                            className="w-full border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-black"
                          >
                            <MessageCircle className="w-4 h-4 mr-2" />
                            DM Gönder
                          </Button>
                        </Link>
                      </div>
                    ) : (
                      <Dialog>
                        <DialogTrigger asChild>
                          <Button 
                            className="w-full bg-accentneongreen text-black hover:bg-accentneongreen/80"
                            onClick={() => setSelectedUser(user)}
                          >
                            <UserPlus className="w-4 h-4 mr-2" />
                            Arkadaşlık İsteği Gönder
                          </Button>
                        </DialogTrigger>
                        <DialogContent className="bg-gray-900 border-gray-700">
                          <DialogHeader>
                            <DialogTitle className="text-white">
                              {selectedUser?.displayName} kullanıcısına arkadaşlık isteği gönder
                            </DialogTitle>
                          </DialogHeader>
                          <div className="space-y-4">
                            <Textarea
                              placeholder="İsteğe bağlı mesaj yazın..."
                              value={requestMessage}
                              onChange={(e) => setRequestMessage(e.target.value)}
                              className="bg-gray-800 border-gray-600 text-white"
                            />
                            <div className="flex space-x-2">
                              <Button
                                onClick={handleSendRequest}
                                disabled={loading}
                                className="flex-1 bg-accentneongreen text-black hover:bg-accentneongreen/80"
                              >
                                İstek Gönder
                              </Button>
                              <Button
                                variant="outline"
                                onClick={() => {
                                  setSelectedUser(null);
                                  setRequestMessage('');
                                }}
                                className="flex-1"
                              >
                                İptal
                              </Button>
                            </div>
                          </div>
                        </DialogContent>
                      </Dialog>
                    )}
                  </CardContent>
                </Card>
              ))}
            </div>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}