import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { BaseCrudService } from '@/integrations';
import { useMember } from '@/integrations';
import { useAdminAccess } from '@/hooks/useAdminAccess';
import { AnimeTitles, MangaTitles, Haberler } from '@/entities';
import { Image } from '@/components/ui/image';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent } from '@/components/ui/card';
import { NotificationBanner } from '@/components/ui/notification-banner';
import { NotificationPopup } from '@/components/ui/notification-popup';
import { AutoNotificationPopup } from '@/components/ui/auto-notification-popup';
import { VideoUploadDialog } from '@/components/ui/video-upload-dialog';
import { Play, BookOpen, Star, Calendar, User, LogOut, Newspaper, FileVideo, Bell, Upload, Download } from 'lucide-react';
import { format } from 'date-fns';
import { tr } from 'date-fns/locale';

// Video type detection
const getVideoType = (url: string): 'youtube' | 'googledrive' | 'invalid' => {
  if (!url) return 'invalid';
  if (url.includes('youtube.com') || url.includes('youtu.be')) {
    return 'youtube';
  }
  if (url.includes('drive.google.com')) {
    return 'googledrive';
  }
  return 'invalid';
};

// Video platform badge component
const VideoPlatformBadge = ({ videoUrl }: { videoUrl: string }) => {
  const videoType = getVideoType(videoUrl);
  
  switch (videoType) {
    case 'youtube':
      return (
        <Badge className="bg-red-500/20 text-red-400 border-red-500/30 text-xs">
          <FileVideo className="w-3 h-3 mr-1" />
          YouTube
        </Badge>
      );
    case 'googledrive':
      return (
        <Badge className="bg-blue-500/20 text-blue-400 border-blue-500/30 text-xs">
          <FileVideo className="w-3 h-3 mr-1" />
          Google Drive
        </Badge>
      );
    default:
      return null;
  }
};

export default function HomePage() {
  const [featuredAnime, setFeaturedAnime] = useState<AnimeTitles[]>([]);
  const [featuredManga, setFeaturedManga] = useState<MangaTitles[]>([]);
  const [latestNews, setLatestNews] = useState<Haberler[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isNotificationPopupOpen, setIsNotificationPopupOpen] = useState(false);
  const [showAutoNotification, setShowAutoNotification] = useState(true);
  const [isUploadDialogOpen, setIsUploadDialogOpen] = useState(false);
  const { member, isAuthenticated, actions } = useMember();
  const { isAdmin } = useAdminAccess();

  useEffect(() => {
    const fetchFeaturedContent = async () => {
      try {
        const [animeResponse, mangaResponse, newsResponse] = await Promise.all([
          BaseCrudService.getAll<AnimeTitles>('animetitles'),
          BaseCrudService.getAll<MangaTitles>('mangatitles'),
          BaseCrudService.getAll<Haberler>('haberler')
        ]);

        const featuredAnimeList = animeResponse.items.filter(anime => anime.isFeatured).slice(0, 3);
        const featuredMangaList = mangaResponse.items.filter(manga => manga.isFeatured).slice(0, 3);
        
        // Sort news by publish date, newest first, and take latest 3
        const sortedNews = newsResponse.items.sort((a, b) => {
          const dateA = new Date(a.publishDate || a._createdDate || 0);
          const dateB = new Date(b.publishDate || b._createdDate || 0);
          return dateB.getTime() - dateA.getTime();
        }).slice(0, 3);

        setFeaturedAnime(featuredAnimeList);
        setFeaturedManga(featuredMangaList);
        setLatestNews(sortedNews);
      } catch (error) {
        console.error('Error fetching featured content:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchFeaturedContent();
  }, []);

  const handleDownloadSite = () => {
    // Create a dummy zip file download
    const link = document.createElement('a');
    link.href = 'data:application/zip;base64,UEsDBAoAAAAAAIdYWVcAAAAAAAAAAAAAAAAJAAAAbWFuZ2l1cy8KUEsDBBQAAAAIAIdYWVcAAAAAAAAAAAAAAABGAAAAbWFuZ2l1cy9SRUFETUUudHh0TUFOR0lVUyAtIEFuaW1lICYgTWFuZ2EgUGxhdGZvcm1hCgpCdSBzaXRlIGluZGlyaWxlYmlsaXIgdmUgbG9rYWwgb2xhcmFrIGt1bGxhbmlsYWJpbGlyLgpQSwECFAAKAAAAAACHWFlXAAAAAAAAAAAAAAAACQAAAAAAAAAAABAA7UEAAAAABW1hbmdpdXMvUEsBAhQAFAAAAAgAh1hZVwAAAAAAAAAAAAAAAEYAAAAAAAAAAAAgAKSBJwAAAABtYW5naXVzL1JFQURNRS50eHRQSwUGAAAAAAIAAgBxAAAAbQAAAAAA';
    link.download = 'mangius-site.zip';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  if (isLoading) {
    return (
      <div className="min-h-screen bg-primary flex items-center justify-center">
        <div className="text-primary-foreground text-xl">Yükleniyor...</div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-primary">
      {/* Navigation */}
      <nav className="w-full px-6 py-4 flex items-center justify-between max-w-[120rem] mx-auto">
        <div className="flex items-center space-x-2">
          <div className="w-8 h-8 bg-accentneongreen rounded-lg flex items-center justify-center">
            <BookOpen className="w-5 h-5 text-primary" />
          </div>
          <span className="font-heading text-xl font-bold text-primary-foreground">Mangius</span>
        </div>
        
        <div className="hidden md:flex items-center space-x-8">
          <Link to="/anime" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Anime
          </Link>
          <Link to="/manga" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Manga
          </Link>
          <Link to="/news" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Haberler
          </Link>
          {isAuthenticated && (
            <Link to="/friends" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
              Arkadaşlar
            </Link>
          )}
        </div>

        <div className="flex items-center space-x-4">
          {/* Site Download Button */}
          <Button
            variant="outline"
            size="sm"
            onClick={handleDownloadSite}
            className="border-accentneongreen/50 text-accentneongreen hover:bg-accentneongreen hover:text-primary transition-all duration-300"
          >
            <Download className="w-4 h-4 mr-2" />
            Siteyi İndir
          </Button>

          {/* Video Upload Button - Admin Only */}
          {isAdmin && (
            <Button
              variant="outline"
              size="sm"
              onClick={() => setIsUploadDialogOpen(true)}
              className="border-accentneongreen/50 text-accentneongreen hover:bg-accentneongreen hover:text-primary transition-all duration-300"
            >
              <Upload className="w-4 h-4 mr-2" />
              Video Yükle
            </Button>
          )}

          {/* Notification Button */}
          <Button
            variant="outline"
            size="sm"
            onClick={() => setIsNotificationPopupOpen(true)}
            className="border-accentneongreen/50 text-accentneongreen hover:bg-accentneongreen hover:text-primary transition-all duration-300 relative"
          >
            <Bell className="w-4 h-4" />
            <span className="sr-only">Bildirimleri Görüntüle</span>
          </Button>

          {!isAuthenticated ? (
            <>
              <Button 
                onClick={actions.login}
                variant="outline" 
                className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary"
              >
                Giriş Yap
              </Button>
              <Button 
                onClick={actions.login}
                className="bg-accentneongreen text-primary hover:bg-accentneongreen/90"
              >
                Başla
              </Button>
            </>
          ) : (
            <>
              <Link to="/profile">
                <Button variant="outline" className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary">
                  <User className="w-4 h-4 mr-2" />
                  {member?.profile?.nickname || member?.contact?.firstName || 'Profil'}
                </Button>
              </Link>
              <Button 
                onClick={actions.logout}
                variant="outline" 
                className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary"
              >
                <LogOut className="w-4 h-4 mr-2" />
                Çıkış Yap
              </Button>
            </>
          )}
        </div>
      </nav>

      {/* Notification Banner */}
      <div className="w-full max-w-[120rem] mx-auto px-6 py-4">
        <NotificationBanner />
      </div>

      {/* Hero Section */}
      <section className="w-full max-w-[120rem] mx-auto px-6 py-20 text-center relative">
        {/* Decorative Elements */}
        <div className="absolute top-20 left-20 w-16 h-16 bg-secondary rounded-2xl opacity-80 hidden lg:block"></div>
        <div className="absolute top-40 right-32 w-12 h-12 bg-accentpeach rounded-full opacity-60 hidden lg:block"></div>
        <div className="absolute bottom-20 left-1/4 w-8 h-8 bg-accentneongreen rounded-lg opacity-70 hidden lg:block"></div>

        <div className="max-w-4xl mx-auto space-y-8">
          <h1 className="font-heading text-6xl md:text-8xl font-black leading-tight">
            <span className="text-primary-foreground">Keşfet</span>{' '}
            <span className="text-accentneongreen">Anime</span>
            <br />
            <span className="text-primary-foreground">& Manga</span>
          </h1>
          
          <p className="font-paragraph text-xl text-primary-foreground/80 max-w-2xl mx-auto">
            Dünyanın en yetenekli yaratıcılarından büyüleyici hikayeler ve çarpıcı sanat eserlerine dalın
          </p>

          <div className="flex flex-col sm:flex-row gap-4 justify-center items-center pt-8">
            <Button 
              onClick={handleDownloadSite}
              size="lg" 
              className="bg-accentneongreen text-primary hover:bg-accentneongreen/90 px-8 py-4 text-lg"
            >
              <Download className="w-5 h-5 mr-2" />
              Siteyi İndir
            </Button>
            
            {!isAuthenticated ? (
              <>
                <Button 
                  onClick={actions.login}
                  size="lg" 
                  variant="outline"
                  className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary px-8 py-4 text-lg"
                >
                  <Play className="w-5 h-5 mr-2" />
                  Okumaya Başla
                </Button>
                <Button size="lg" variant="outline" className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary px-8 py-4 text-lg">
                  Koleksiyonu Gözat
                </Button>
              </>
            ) : (
              <>
                <Link to="/anime">
                  <Button size="lg" variant="outline" className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary px-8 py-4 text-lg">
                    <Play className="w-5 h-5 mr-2" />
                    Okumaya Başla
                  </Button>
                </Link>
                <Link to="/profile">
                  <Button size="lg" variant="outline" className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary px-8 py-4 text-lg">
                    <User className="w-5 h-5 mr-2" />
                    Profilim
                  </Button>
                </Link>
              </>
            )}
          </div>
        </div>
      </section>

      {/* Featured Anime Section */}
      <section className="w-full max-w-[120rem] mx-auto px-6 py-16">
        <div className="flex items-center justify-between mb-12">
          <div>
            <h2 className="font-heading text-4xl font-bold text-primary-foreground mb-2">
              Öne Çıkan <span className="text-accentneongreen">Anime</span>
            </h2>
            <p className="font-paragraph text-primary-foreground/70">
              En popüler anime serilerini keşfedin
            </p>
          </div>
          <Link to="/anime">
            <Button variant="outline" className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary">
              Tüm Animeleri Gör
            </Button>
          </Link>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
          {featuredAnime.map((anime) => (
            <Link key={anime._id} to={`/anime/${anime._id}`}>
              <Card className="bg-secondary border-0 hover:bg-secondary/80 transition-all duration-300 group overflow-hidden">
                <div className="aspect-[3/4] relative overflow-hidden">
                  {(anime.posterImageUrl || anime.posterImage) ? (
                    <Image
                      src={anime.posterImageUrl || anime.posterImage || ''}
                      alt={anime.title || 'Anime posteri'}
                      className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                      width={300}
                    />
                  ) : (
                    <div className="w-full h-full bg-secondary-foreground/10 flex items-center justify-center">
                      <Play className="w-16 h-16 text-secondary-foreground/30" />
                    </div>
                  )}
                  <div className="absolute top-4 left-4">
                    <Badge className="bg-accentneongreen text-primary">
                      <Star className="w-3 h-3 mr-1" />
                      Öne Çıkan
                    </Badge>
                  </div>
                  {anime.releaseYear && (
                    <div className="absolute top-4 right-4">
                      <Badge variant="secondary" className="bg-primary/80 text-primary-foreground">
                        <Calendar className="w-3 h-3 mr-1" />
                        {anime.releaseYear}
                      </Badge>
                    </div>
                  )}
                </div>
                <CardContent className="p-6">
                  <h3 className="font-heading text-xl font-bold text-secondary-foreground mb-2 line-clamp-1">
                    {anime.title}
                  </h3>
                  <p className="font-paragraph text-secondary-foreground/70 text-sm mb-3 line-clamp-2">
                    {anime.synopsis}
                  </p>
                  <div className="flex items-center justify-between text-sm">
                    <span className="font-paragraph text-accentneongreen font-medium">
                      {anime.genre}
                    </span>
                    {anime.numberOfEpisodes && (
                      <span className="font-paragraph text-secondary-foreground/60">
                        {anime.numberOfEpisodes} bölüm
                      </span>
                    )}
                  </div>
                </CardContent>
              </Card>
            </Link>
          ))}
        </div>
      </section>

      {/* Latest News Section */}
      {latestNews.length > 0 && (
        <section className="w-full max-w-[120rem] mx-auto px-6 py-16">
          <div className="flex items-center justify-between mb-12">
            <div>
              <h2 className="font-heading text-4xl font-bold text-primary-foreground mb-2">
                Son <span className="text-accentneongreen">Haberler</span>
              </h2>
              <p className="font-paragraph text-primary-foreground/70">
                Anime ve manga dünyasından en güncel haberler
              </p>
            </div>
            <Link to="/news">
              <Button variant="outline" className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary">
                Tüm Haberleri Gör
              </Button>
            </Link>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {latestNews.map((news) => (
              <Link key={news._id} to={`/news/${news._id}`}>
                <Card className="bg-secondary border-0 hover:bg-secondary/80 transition-all duration-300 group overflow-hidden">
                  {news.mainImage && (
                    <div className="aspect-[16/9] relative overflow-hidden">
                      <Image
                        src={news.mainImage}
                        alt={news.title || 'Haber görseli'}
                        className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                        width={400}
                      />
                      <div className="absolute top-4 left-4">
                        <Badge className="bg-accentneongreen text-primary">
                          <Newspaper className="w-3 h-3 mr-1" />
                          Haber
                        </Badge>
                      </div>
                    </div>
                  )}
                  <CardContent className="p-6">
                    <h3 className="font-heading text-xl font-bold text-secondary-foreground mb-2 line-clamp-2">
                      {news.title}
                    </h3>
                    {news.excerpt && (
                      <p className="font-paragraph text-secondary-foreground/70 text-sm mb-3 line-clamp-2">
                        {news.excerpt}
                      </p>
                    )}
                    <div className="flex items-center justify-between text-sm">
                      <div className="flex items-center gap-2">
                        {news.author && (
                          <span className="font-paragraph text-accentneongreen font-medium">
                            {news.author}
                          </span>
                        )}
                        {news.videoUrl && (
                          <VideoPlatformBadge videoUrl={news.videoUrl} />
                        )}
                      </div>
                      {news.publishDate && (
                        <span className="font-paragraph text-secondary-foreground/60">
                          {format(new Date(news.publishDate), 'dd MMM yyyy', { locale: tr })}
                        </span>
                      )}
                    </div>
                  </CardContent>
                </Card>
              </Link>
            ))}
          </div>
        </section>
      )}

      {/* Featured Manga Section */}
      <section className="w-full max-w-[120rem] mx-auto px-6 py-16">
        <div className="flex items-center justify-between mb-12">
          <div>
            <h2 className="font-heading text-4xl font-bold text-primary-foreground mb-2">
              Öne Çıkan <span className="text-accentneongreen">Manga</span>
            </h2>
            <p className="font-paragraph text-primary-foreground/70">
              Büyüleyici manga hikayelerini keşfedin
            </p>
          </div>
          <Link to="/manga">
            <Button variant="outline" className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary">
              Tüm Mangaları Gör
            </Button>
          </Link>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
          {featuredManga.map((manga) => (
            <Link key={manga._id} to={`/manga/${manga._id}`}>
              <Card className="bg-secondary border-0 hover:bg-secondary/80 transition-all duration-300 group overflow-hidden">
                <div className="aspect-[3/4] relative overflow-hidden">
                  {(manga.coverImageUrl || manga.coverImage) ? (
                    <Image
                      src={manga.coverImageUrl || manga.coverImage || ''}
                      alt={manga.title || 'Manga kapağı'}
                      className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                      width={300}
                    />
                  ) : (
                    <div className="w-full h-full bg-secondary-foreground/10 flex items-center justify-center">
                      <BookOpen className="w-16 h-16 text-secondary-foreground/30" />
                    </div>
                  )}
                  <div className="absolute top-4 left-4">
                    <Badge className="bg-accentneongreen text-primary">
                      <Star className="w-3 h-3 mr-1" />
                      Öne Çıkan
                    </Badge>
                  </div>
                </div>
                <CardContent className="p-6">
                  <h3 className="font-heading text-xl font-bold text-secondary-foreground mb-2 line-clamp-1">
                    {manga.title}
                  </h3>
                  <p className="font-paragraph text-secondary-foreground/70 text-sm mb-3 line-clamp-2">
                    {manga.synopsis}
                  </p>
                  <div className="flex items-center justify-between text-sm">
                    <span className="font-paragraph text-accentneongreen font-medium">
                      {manga.genre}
                    </span>
                    {manga.numberOfChapters && (
                      <span className="font-paragraph text-secondary-foreground/60">
                        {manga.numberOfChapters} bölüm
                      </span>
                    )}
                  </div>
                </CardContent>
              </Card>
            </Link>
          ))}
        </div>
      </section>

      {/* Call to Action Section */}
      <section className="w-full max-w-[120rem] mx-auto px-6 py-20">
        <div className="bg-secondary rounded-3xl p-12 text-center relative overflow-hidden">
          {/* Decorative Elements */}
          <div className="absolute top-8 left-8 w-20 h-20 bg-accentneongreen/20 rounded-2xl"></div>
          <div className="absolute bottom-8 right-8 w-16 h-16 bg-accentpeach/30 rounded-full"></div>
          
          <div className="relative z-10 max-w-3xl mx-auto">
            <h2 className="font-heading text-5xl font-black text-secondary-foreground mb-6">
              <span className="text-accentneongreen">Topluluğa</span> Katıl
            </h2>
            <p className="font-paragraph text-xl text-secondary-foreground/80 mb-8">
              Anime ve manga tutkunlarıyla bağlantı kurun, favorilerinizi paylaşın ve yeni hikayeler keşfedin
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              {!isAuthenticated ? (
                <>
                  <Button 
                    onClick={actions.login}
                    size="lg" 
                    className="bg-accentneongreen text-primary hover:bg-accentneongreen/90 px-8 py-4"
                  >
                    Hesap Oluştur
                  </Button>
                  <Button size="lg" variant="outline" className="border-secondary-foreground text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary px-8 py-4">
                    Daha Fazla Bilgi
                  </Button>
                </>
              ) : (
                <>
                  <Link to="/profile">
                    <Button size="lg" className="bg-accentneongreen text-primary hover:bg-accentneongreen/90 px-8 py-4">
                      <User className="w-5 h-5 mr-2" />
                      Profilimi Görüntüle
                    </Button>
                  </Link>
                  <Link to="/anime">
                    <Button size="lg" variant="outline" className="border-secondary-foreground text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary px-8 py-4">
                      Koleksiyonu Keşfet
                    </Button>
                  </Link>
                </>
              )}
            </div>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="w-full bg-primary border-t border-primary-foreground/10">
        <div className="max-w-[120rem] mx-auto px-6 py-16">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            <div className="space-y-4">
              <div className="flex items-center space-x-2">
                <div className="w-8 h-8 bg-accentneongreen rounded-lg flex items-center justify-center">
                  <BookOpen className="w-5 h-5 text-primary" />
                </div>
                <span className="font-heading text-xl font-bold text-primary-foreground">Mangius</span>
              </div>
              <p className="font-paragraph text-primary-foreground/70">
                Dünyanın en iyi anime ve manga içeriklerine açılan kapınız.
              </p>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Gözat</h3>
              <div className="space-y-2">
                <Link to="/anime" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Anime Koleksiyonu
                </Link>
                <Link to="/manga" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Manga Kütüphanesi
                </Link>
                <Link to="/featured" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Öne Çıkan İçerik
                </Link>
              </div>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Topluluk</h3>
              <div className="space-y-2">
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Forumlar
                </a>
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  İncelemeler
                </a>
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Öneriler
                </a>
              </div>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Destek</h3>
              <div className="space-y-2">
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Yardım Merkezi
                </a>
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  İletişim
                </a>
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Gizlilik Politikası
                </a>
                <button 
                  onClick={handleDownloadSite}
                  className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors text-left"
                >
                  Siteyi İndir
                </button>
              </div>
            </div>
          </div>
          
          <div className="border-t border-primary-foreground/10 mt-12 pt-8 text-center">
            <p className="font-paragraph text-primary-foreground/60">
              © 2024 Mangius. Tüm hakları saklıdır.
            </p>
          </div>
        </div>
      </footer>

      {/* Notification Popup */}
      <NotificationPopup 
        isOpen={isNotificationPopupOpen} 
        onClose={() => setIsNotificationPopupOpen(false)} 
      />

      {/* Auto Notification Popup */}
      {showAutoNotification && (
        <AutoNotificationPopup 
          onClose={() => setShowAutoNotification(false)} 
        />
      )}

      {/* Video Upload Dialog - Admin Only */}
      {isAdmin && (
        <VideoUploadDialog 
          isOpen={isUploadDialogOpen} 
          onClose={() => setIsUploadDialogOpen(false)} 
        />
      )}
    </div>
  );
}