import { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import { BaseCrudService } from '@/integrations';
import { useMember } from '@/integrations';
import { MangaTitles, MangaVolumes } from '@/entities';
import { useFavorites } from '@/hooks/useFavorites';
import { useAdminAccess } from '@/hooks/useAdminAccess';
import { CommentSection } from '@/components/ui/comment-section';
import { MangaPreview } from '@/components/ui/manga-preview';
import { VolumePreview } from '@/components/ui/volume-preview';
import { Image } from '@/components/ui/image';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent } from '@/components/ui/card';
import { BookOpen, ArrowLeft, Book, Calendar, User, Tag, FileText, LogOut, Heart, Settings, Eye, ExternalLink, Play, Upload, Plus } from 'lucide-react';
import { MangaChapterUploadDialog } from '@/components/ui/manga-chapter-upload-dialog';
import { format } from 'date-fns';

export default function MangaDetailPage() {
  const { id } = useParams<{ id: string }>();
  const [manga, setManga] = useState<MangaTitles | null>(null);
  const [volumes, setVolumes] = useState<MangaVolumes[]>([]);
  const [relatedManga, setRelatedManga] = useState<MangaTitles[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [showPreview, setShowPreview] = useState(false);
  const [selectedVolume, setSelectedVolume] = useState<MangaVolumes | null>(null);
  const [showVolumePreview, setShowVolumePreview] = useState(false);
  const [isChapterUploadDialogOpen, setIsChapterUploadDialogOpen] = useState(false);
  const { member, isAuthenticated, actions } = useMember();
  
  // Admin access check
  const { isAdmin } = useAdminAccess();
  
  // Favorites hook
  const { addToFavorites, removeFromFavorites, isFavorite } = useFavorites(
    member?.loginEmail
  );

  const handleFavoriteToggle = async () => {
    if (!isAuthenticated || !manga) {
      actions.login();
      return;
    }

    const isCurrentlyFavorite = isFavorite(manga._id, 'manga');
    
    if (isCurrentlyFavorite) {
      await removeFromFavorites(manga._id, 'manga');
    } else {
      await addToFavorites(manga._id, 'manga');
    }
  };

  const handleChapterAdded = () => {
    // Refresh volumes/chapters list after new chapter is added
    const fetchVolumes = async () => {
      if (!id) return;
      
      try {
        const volumesResponse = await BaseCrudService.getAll<MangaVolumes>('mangavolumes');
        // Filter volumes by mangaSeriesId to show only volumes belonging to this manga
        const mangaVolumes = volumesResponse.items
          .filter(volume => volume.mangaSeriesId === id)
          .sort((a, b) => (a.volumeNumber || 0) - (b.volumeNumber || 0));
        setVolumes(mangaVolumes);
      } catch (error) {
        console.error('Error fetching volumes:', error);
      }
    };
    fetchVolumes();
  };

  const handleVolumeClick = (volume: MangaVolumes) => {
    setSelectedVolume(volume);
    setShowVolumePreview(true);
  };

  useEffect(() => {
    const fetchManga = async () => {
      if (!id) return;
      
      try {
        const mangaData = await BaseCrudService.getById<MangaTitles>('mangatitles', id);
        setManga(mangaData);

        // Fetch volumes for this manga
        const volumesResponse = await BaseCrudService.getAll<MangaVolumes>('mangavolumes');
        const mangaVolumes = volumesResponse.items
          .filter(volume => volume.mangaSeriesId === id)
          .sort((a, b) => (a.volumeNumber || 0) - (b.volumeNumber || 0));
        setVolumes(mangaVolumes);

        // Fetch related manga (same genre)
        if (mangaData?.genre) {
          const allMangaResponse = await BaseCrudService.getAll<MangaTitles>('mangatitles');
          const related = allMangaResponse.items
            .filter(item => item.genre === mangaData.genre && item._id !== mangaData._id)
            .slice(0, 4);
          setRelatedManga(related);
        }
      } catch (error) {
        console.error('Error fetching manga:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchManga();
  }, [id]);

  if (isLoading) {
    return (
      <div className="min-h-screen bg-primary flex items-center justify-center">
        <div className="text-primary-foreground text-xl">Manga detayları yükleniyor...</div>
      </div>
    );
  }

  if (!manga) {
    return (
      <div className="min-h-screen bg-primary flex items-center justify-center">
        <div className="text-center">
          <h2 className="font-heading text-2xl font-bold text-primary-foreground mb-4">
            Manga bulunamadı
          </h2>
          <Link to="/manga">
            <Button className="bg-accentneongreen text-primary hover:bg-accentneongreen/90">
              Manga Kütüphanesine Dön
            </Button>
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-primary">
      {/* Navigation */}
      <nav className="w-full px-6 py-4 flex items-center justify-between max-w-[120rem] mx-auto border-b border-primary-foreground/10">
        <Link to="/" className="flex items-center space-x-2">
          <div className="w-8 h-8 bg-accentneongreen rounded-lg flex items-center justify-center">
            <BookOpen className="w-5 h-5 text-primary" />
          </div>
          <span className="font-heading text-xl font-bold text-primary-foreground">Mangius</span>
        </Link>
        
        <div className="hidden md:flex items-center space-x-8">
          <Link to="/anime" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Anime
          </Link>
          <Link to="/manga" className="font-paragraph text-accentneongreen font-medium">
            Manga
          </Link>
          {isAuthenticated && (
            <Link to="/friends" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
              Arkadaşlar
            </Link>
          )}
          <Link to="/" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Ana Sayfa
          </Link>
        </div>

        <div className="flex items-center space-x-4">
          {!isAuthenticated ? (
            <Button 
              onClick={actions.login}
              variant="outline" 
              className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary"
            >
              Giriş Yap
            </Button>
          ) : (
            <>
              {isAdmin && (
                <Link to="/admin">
                  <Button variant="outline" className="border-secondary text-secondary hover:bg-secondary hover:text-secondary-foreground">
                    <Settings className="w-4 h-4 mr-2" />
                    Admin
                  </Button>
                </Link>
              )}
              <Link to="/profile">
                <Button variant="outline" className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary">
                  <User className="w-4 h-4 mr-2" />
                  {member?.profile?.nickname || 'Profil'}
                </Button>
              </Link>
              <Button 
                onClick={actions.logout}
                variant="outline" 
                className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary"
              >
                <LogOut className="w-4 h-4 mr-2" />
                Çıkış Yap
              </Button>
            </>
          )}
        </div>
      </nav>

      {/* Back Button */}
      <div className="w-full max-w-[120rem] mx-auto px-6 py-6">
        <Link to="/manga">
          <Button variant="outline" className="border-primary-foreground/30 text-primary-foreground hover:bg-primary-foreground hover:text-primary">
            <ArrowLeft className="w-4 h-4 mr-2" />
            Manga Kütüphanesine Dön
          </Button>
        </Link>
      </div>

      {/* Manga Detail Section */}
      <section className="w-full max-w-[120rem] mx-auto px-6 pb-16">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
          {/* Cover and Quick Info */}
          <div className="lg:col-span-1">
            <div className="sticky top-8">
              <div className="aspect-[3/4] relative overflow-hidden rounded-2xl mb-6">
                {(manga.coverImageUrl || manga.coverImage) ? (
                  <Image
                    src={manga.coverImageUrl || manga.coverImage || ''}
                    alt={manga.title || 'Manga kapağı'}
                    className="w-full h-full object-cover"
                    width={400}
                  />
                ) : (
                  <div className="w-full h-full bg-secondary flex items-center justify-center rounded-2xl">
                    <Book className="w-16 h-16 text-secondary-foreground/40" />
                  </div>
                )}
                
                {manga.isFeatured && (
                  <div className="absolute top-4 left-4">
                    <Badge className="bg-accentneongreen text-primary">
                      Öne Çıkan
                    </Badge>
                  </div>
                )}
              </div>

              <div className="space-y-4">
                <Button 
                  size="lg" 
                  className="w-full bg-accentneongreen text-primary hover:bg-accentneongreen/90"
                  onClick={() => setShowPreview(true)}
                >
                  <Eye className="w-5 h-5 mr-2" />
                  Genel Önizleme
                </Button>
                
                <Button 
                  size="lg" 
                  variant="outline" 
                  className={`w-full ${
                    manga && isFavorite(manga._id, 'manga')
                      ? 'border-red-500 text-red-500 hover:bg-red-500 hover:text-white'
                      : 'border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary'
                  }`}
                  onClick={handleFavoriteToggle}
                >
                  <Heart className={`w-5 h-5 mr-2 ${manga && isFavorite(manga._id, 'manga') ? 'fill-current' : ''}`} />
                  {manga && isFavorite(manga._id, 'manga') ? 'Favorilerden Çıkar' : 'Favorilere Ekle'}
                </Button>
              </div>
            </div>
          </div>

          {/* Main Content */}
          <div className="lg:col-span-2 space-y-8">
            {/* Title and Basic Info */}
            <div>
              <h1 className="font-heading text-4xl md:text-5xl font-black text-primary-foreground mb-4">
                {manga.title}
              </h1>
              
              <div className="flex flex-wrap gap-4 mb-6">
                {manga.genre && (
                  <div className="flex items-center space-x-2">
                    <Tag className="w-4 h-4 text-accentneongreen" />
                    <span className="font-paragraph text-accentneongreen font-medium">
                      {manga.genre}
                    </span>
                  </div>
                )}
                
                {manga.publicationDate && (
                  <div className="flex items-center space-x-2">
                    <Calendar className="w-4 h-4 text-primary-foreground/60" />
                    <span className="font-paragraph text-primary-foreground/60">
                      {format(new Date(manga.publicationDate), 'MMMM yyyy')}
                    </span>
                  </div>
                )}
                
                {manga.numberOfChapters && (
                  <div className="flex items-center space-x-2">
                    <FileText className="w-4 h-4 text-primary-foreground/60" />
                    <span className="font-paragraph text-primary-foreground/60">
                      {manga.numberOfChapters} Bölüm
                    </span>
                  </div>
                )}
                
                {manga.author && (
                  <div className="flex items-center space-x-2">
                    <User className="w-4 h-4 text-primary-foreground/60" />
                    <span className="font-paragraph text-primary-foreground/60">
                      {manga.author}
                    </span>
                  </div>
                )}
              </div>
            </div>

            {/* Synopsis */}
            <div className="bg-secondary rounded-2xl p-8">
              <h2 className="font-heading text-2xl font-bold text-secondary-foreground mb-4">
                Özet
              </h2>
              <p className="font-paragraph text-secondary-foreground/80 leading-relaxed text-lg">
                {manga.synopsis || 'Bu manga için özet mevcut değil.'}
              </p>
            </div>

            {/* Volumes Section */}
            {volumes.length > 0 && (
              <div className="bg-secondary rounded-2xl p-8">
                <h2 className="font-heading text-2xl font-bold text-secondary-foreground mb-6">
                  Ciltler
                </h2>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                  {volumes.map((volume) => (
                    <Card 
                      key={volume._id} 
                      className="bg-primary border-0 hover:bg-primary/80 transition-all duration-300 cursor-pointer group"
                      onClick={() => handleVolumeClick(volume)}
                    >
                      <CardContent className="p-4">
                        <div className="flex items-center space-x-3">
                          <div className="w-12 h-16 bg-accentneongreen/20 rounded-lg flex items-center justify-center flex-shrink-0">
                            <Book className="w-6 h-6 text-accentneongreen" />
                          </div>
                          <div className="flex-1 min-w-0">
                            <h3 className="font-heading text-sm font-bold text-primary-foreground truncate">
                              {volume.volumeTitle || `Cilt ${volume.volumeNumber}`}
                            </h3>
                            {volume.volumeNumber && (
                              <p className="font-paragraph text-primary-foreground/60 text-xs">
                                Cilt {volume.volumeNumber}
                              </p>
                            )}
                          </div>
                          <Play className="w-4 h-4 text-accentneongreen group-hover:scale-110 transition-transform" />
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </div>
            )}

            {/* Additional Info */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <Card className="bg-secondary border-0">
                <CardContent className="p-6">
                  <h3 className="font-heading text-lg font-bold text-secondary-foreground mb-4">
                    Detaylar
                  </h3>
                  <div className="space-y-3">
                    <div className="flex justify-between">
                      <span className="font-paragraph text-secondary-foreground/70">Durum:</span>
                      <span className="font-paragraph text-secondary-foreground font-medium">
                        {manga.isFeatured ? 'Öne Çıkan' : 'Mevcut'}
                      </span>
                    </div>
                    {manga.publicationDate && (
                      <div className="flex justify-between">
                        <span className="font-paragraph text-secondary-foreground/70">Yayınlandı:</span>
                        <span className="font-paragraph text-secondary-foreground font-medium">
                          {format(new Date(manga.publicationDate), 'yyyy')}
                        </span>
                      </div>
                    )}
                    {manga.numberOfChapters && (
                      <div className="flex justify-between">
                        <span className="font-paragraph text-secondary-foreground/70">Bölümler:</span>
                        <span className="font-paragraph text-secondary-foreground font-medium">
                          {manga.numberOfChapters}
                        </span>
                      </div>
                    )}
                    {volumes.length > 0 && (
                      <div className="flex justify-between">
                        <span className="font-paragraph text-secondary-foreground/70">Ciltler:</span>
                        <div className="flex items-center gap-2">
                          <span className="font-paragraph text-accentneongreen font-medium">
                            {volumes.length} Cilt
                          </span>
                          {isAdmin && (
                            <Button
                              onClick={() => setIsChapterUploadDialogOpen(true)}
                              size="sm"
                              variant="outline"
                              className="border-accentneongreen/30 text-accentneongreen hover:bg-accentneongreen hover:text-primary text-xs px-2 py-1 h-6"
                            >
                              <Plus className="w-3 h-3 mr-1" />
                              Bölüm Ekle
                            </Button>
                          )}
                        </div>
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-secondary border-0">
                <CardContent className="p-6">
                  <h3 className="font-heading text-lg font-bold text-secondary-foreground mb-4">
                    Yayın
                  </h3>
                  <div className="space-y-3">
                    {manga.author && (
                      <div className="flex justify-between">
                        <span className="font-paragraph text-secondary-foreground/70">Yazar:</span>
                        <span className="font-paragraph text-secondary-foreground font-medium">
                          {manga.author}
                        </span>
                      </div>
                    )}
                    {manga.genre && (
                      <div className="flex justify-between">
                        <span className="font-paragraph text-secondary-foreground/70">Tür:</span>
                        <span className="font-paragraph text-accentneongreen font-medium">
                          {manga.genre}
                        </span>
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Comments Section */}
            {manga && (
              <CommentSection 
                contentId={manga._id} 
                contentType="manga" 
              />
            )}
          </div>
        </div>

        {/* Related Manga */}
        {relatedManga.length > 0 && (
          <div className="mt-16">
            <h2 className="font-heading text-3xl font-bold text-primary-foreground mb-8">
              <span className="text-accentneongreen">{manga.genre}</span> Türünde Daha Fazla
            </h2>
            
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
              {relatedManga.map((relatedItem) => (
                <Link key={relatedItem._id} to={`/manga/${relatedItem._id}`}>
                  <Card className="bg-secondary border-0 hover:bg-secondary/80 transition-all duration-300 group overflow-hidden">
                    <div className="aspect-[3/4] relative overflow-hidden">
                      {(relatedItem.coverImageUrl || relatedItem.coverImage) ? (
                        <Image
                          src={relatedItem.coverImageUrl || relatedItem.coverImage || ''}
                          alt={relatedItem.title || 'Manga kapağı'}
                          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                          width={250}
                        />
                      ) : (
                        <div className="w-full h-full bg-primary/20 flex items-center justify-center">
                          <Book className="w-8 h-8 text-secondary-foreground/40" />
                        </div>
                      )}
                    </div>
                    <CardContent className="p-4">
                      <h3 className="font-heading text-sm font-bold text-secondary-foreground line-clamp-2">
                        {relatedItem.title}
                      </h3>
                      {relatedItem.author && (
                        <p className="font-paragraph text-secondary-foreground/60 text-xs mt-1">
                          {relatedItem.author} tarafından
                        </p>
                      )}
                    </CardContent>
                  </Card>
                </Link>
              ))}
            </div>
          </div>
        )}
      </section>

      {/* Manga Preview Modal */}
      {manga && (
        <MangaPreview 
          manga={manga}
          isOpen={showPreview}
          onClose={() => setShowPreview(false)}
        />
      )}

      {/* Volume Preview Modal */}
      {selectedVolume && (
        <VolumePreview 
          volume={selectedVolume}
          isOpen={showVolumePreview}
          onClose={() => setShowVolumePreview(false)}
        />
      )}

      {/* Footer */}
      <footer className="w-full bg-primary border-t border-primary-foreground/10">
        <div className="max-w-[120rem] mx-auto px-6 py-12">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="space-y-4">
              <div className="flex items-center space-x-2">
                <div className="w-8 h-8 bg-accentneongreen rounded-lg flex items-center justify-center">
                  <BookOpen className="w-5 h-5 text-primary" />
                </div>
                <span className="font-heading text-xl font-bold text-primary-foreground">Mangius</span>
              </div>
              <p className="font-paragraph text-primary-foreground/70">
                En iyi anime ve manga içeriklerine açılan kapınız.
              </p>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Gözat</h3>
              <div className="space-y-2">
                <Link to="/anime" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Anime Koleksiyonu
                </Link>
                <Link to="/manga" className="block font-paragraph text-accentneongreen">
                  Manga Kütüphanesi
                </Link>
                <Link to="/" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Ana Sayfa
                </Link>
              </div>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Destek</h3>
              <div className="space-y-2">
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Yardım Merkezi
                </a>
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  İletişim
                </a>
              </div>
            </div>
          </div>
          
          <div className="border-t border-primary-foreground/10 mt-8 pt-8 text-center">
            <p className="font-paragraph text-primary-foreground/60">
              © 2024 Mangius. Tüm hakları saklıdır.
            </p>
          </div>
        </div>
      </footer>

      {/* Chapter Upload Dialog */}
      <MangaChapterUploadDialog
        isOpen={isChapterUploadDialogOpen}
        onClose={() => setIsChapterUploadDialogOpen(false)}
        mangaId={id || ''}
        mangaTitle={manga?.title}
        onChapterAdded={handleChapterAdded}
      />
    </div>
  );
}