import React, { useState, useEffect, useRef } from 'react';
import { useParams, Link } from 'react-router-dom';
import { useMember } from '@/integrations';
import { useFriendship } from '@/hooks/useFriendship';
import { BaseCrudService } from '@/integrations';
import { KullancProfilleri } from '@/entities';
import type { Member } from '@/types/member';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Send, ArrowLeft, MessageCircle } from 'lucide-react';

export default function MessagesPage() {
  const { userId } = useParams<{ userId: string }>();
  const { member }: { member: Member | null } = useMember();
  const { messages, sendMessage, loadMessages, markAsRead, areFriends } = useFriendship();
  
  const [otherUserProfile, setOtherUserProfile] = useState<KullancProfilleri | null>(null);
  const [newMessage, setNewMessage] = useState('');
  const [loading, setLoading] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if (userId) {
      loadUserProfile();
      loadMessages(userId);
    }
  }, [userId]);

  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  const loadUserProfile = async () => {
    if (!userId) return;
    
    try {
      const { items } = await BaseCrudService.getAll<KullancProfilleri>('kullancprofilleri');
      const profile = items.find(user => user.memberId === userId);
      setOtherUserProfile(profile || null);
    } catch (error) {
      console.error('Kullanıcı profili yüklenemedi:', error);
    }
  };

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  const handleSendMessage = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!newMessage.trim() || !userId || loading) return;

    setLoading(true);
    try {
      await sendMessage(userId, newMessage.trim());
      setNewMessage('');
    } catch (error) {
      console.error('Mesaj gönderilemedi:', error);
    } finally {
      setLoading(false);
    }
  };

  const formatTime = (date: Date | string | undefined) => {
    if (!date) return '';
    return new Date(date).toLocaleTimeString('tr-TR', {
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const formatDate = (date: Date | string | undefined) => {
    if (!date) return '';
    return new Date(date).toLocaleDateString('tr-TR');
  };

  // Arkadaş değilse erişimi engelle
  if (userId && !areFriends(userId)) {
    return (
      <div className="min-h-screen bg-black text-white">
        <header className="bg-black border-b border-gray-800 sticky top-0 z-50">
          <div className="max-w-[120rem] mx-auto px-6 py-4">
            <div className="flex items-center justify-between">
              <Link to="/" className="text-2xl font-bold text-accentneongreen">
                AnimeHub
              </Link>
              <nav className="flex items-center space-x-8">
                <Link to="/" className="hover:text-accentneongreen transition-colors">Ana Sayfa</Link>
                <Link to="/anime" className="hover:text-accentneongreen transition-colors">Anime</Link>
                <Link to="/manga" className="hover:text-accentneongreen transition-colors">Manga</Link>
                <Link to="/friends" className="hover:text-accentneongreen transition-colors">Arkadaşlar</Link>
                <Link to="/profile" className="hover:text-accentneongreen transition-colors">Profil</Link>
              </nav>
            </div>
          </div>
        </header>

        <div className="max-w-[120rem] mx-auto px-6 py-8">
          <Card className="bg-gray-900 border-gray-800">
            <CardContent className="p-8 text-center">
              <MessageCircle className="w-16 h-16 mx-auto mb-4 text-gray-600" />
              <h3 className="text-xl font-semibold mb-2">Mesajlaşma İzni Yok</h3>
              <p className="text-gray-400 mb-4">
                Bu kullanıcıyla mesajlaşabilmek için önce arkadaş olmanız gerekiyor.
              </p>
              <Link to="/friends">
                <Button className="bg-accentneongreen text-black hover:bg-accentneongreen/80">
                  Arkadaşlar Sayfasına Git
                </Button>
              </Link>
            </CardContent>
          </Card>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-black text-white flex flex-col">
      {/* Header */}
      <header className="bg-black border-b border-gray-800 sticky top-0 z-50">
        <div className="max-w-[120rem] mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <Link to="/" className="text-2xl font-bold text-accentneongreen">
              AnimeHub
            </Link>
            <nav className="flex items-center space-x-8">
              <Link to="/" className="hover:text-accentneongreen transition-colors">Ana Sayfa</Link>
              <Link to="/anime" className="hover:text-accentneongreen transition-colors">Anime</Link>
              <Link to="/manga" className="hover:text-accentneongreen transition-colors">Manga</Link>
              <Link to="/friends" className="hover:text-accentneongreen transition-colors">Arkadaşlar</Link>
              <Link to="/profile" className="hover:text-accentneongreen transition-colors">Profil</Link>
            </nav>
          </div>
        </div>
      </header>

      {/* Chat Header */}
      <div className="bg-gray-900 border-b border-gray-800 px-6 py-4">
        <div className="max-w-[120rem] mx-auto flex items-center space-x-4">
          <Link to="/friends">
            <Button variant="ghost" size="sm" className="text-gray-400 hover:text-white">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Geri
            </Button>
          </Link>
          <Avatar className="w-10 h-10">
            <AvatarImage src={otherUserProfile?.profilePicture} />
            <AvatarFallback className="bg-accentneongreen text-black">
              {otherUserProfile?.displayName?.charAt(0) || 'U'}
            </AvatarFallback>
          </Avatar>
          <div>
            <h2 className="font-semibold">{otherUserProfile?.displayName || 'Bilinmeyen Kullanıcı'}</h2>
            <p className="text-sm text-gray-400">Aktif</p>
          </div>
        </div>
      </div>

      {/* Messages Area */}
      <div className="flex-1 flex flex-col max-w-[120rem] mx-auto w-full">
        <ScrollArea className="flex-1 px-6 py-4">
          <div className="space-y-4">
            {messages.length === 0 ? (
              <div className="text-center py-8">
                <MessageCircle className="w-16 h-16 mx-auto mb-4 text-gray-600" />
                <p className="text-gray-400">Henüz mesaj yok. İlk mesajı gönderin!</p>
              </div>
            ) : (
              messages.map((message, index) => {
                const isOwnMessage = message.senderId === member?._id;
                const showDate = index === 0 || 
                  formatDate(messages[index - 1]?.sentAt) !== formatDate(message.sentAt);

                return (
                  <div key={message._id}>
                    {showDate && (
                      <div className="text-center py-2">
                        <span className="text-xs text-gray-500 bg-gray-800 px-3 py-1 rounded-full">
                          {formatDate(message.sentAt)}
                        </span>
                      </div>
                    )}
                    <div className={`flex ${isOwnMessage ? 'justify-end' : 'justify-start'}`}>
                      <div className={`max-w-xs lg:max-w-md px-4 py-2 rounded-lg ${
                        isOwnMessage 
                          ? 'bg-accentneongreen text-black' 
                          : 'bg-gray-800 text-white'
                      }`}>
                        <p className="text-sm">{message.messageContent}</p>
                        <p className={`text-xs mt-1 ${
                          isOwnMessage ? 'text-black/70' : 'text-gray-400'
                        }`}>
                          {formatTime(message.sentAt)}
                        </p>
                      </div>
                    </div>
                  </div>
                );
              })
            )}
            <div ref={messagesEndRef} />
          </div>
        </ScrollArea>

        {/* Message Input */}
        <div className="border-t border-gray-800 p-6">
          <form onSubmit={handleSendMessage} className="flex space-x-4">
            <Input
              value={newMessage}
              onChange={(e) => setNewMessage(e.target.value)}
              placeholder="Mesajınızı yazın..."
              className="flex-1 bg-gray-900 border-gray-700 text-white"
              disabled={loading}
            />
            <Button
              type="submit"
              disabled={loading || !newMessage.trim()}
              className="bg-accentneongreen text-black hover:bg-accentneongreen/80"
            >
              <Send className="w-4 h-4" />
            </Button>
          </form>
        </div>
      </div>
    </div>
  );
}