import React, { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import { BaseCrudService } from '@/integrations';
import { Haberler } from '@/entities';
import { Button } from '@/components/ui/button';
import { LoadingSpinner } from '@/components/ui/loading-spinner';
import { Image } from '@/components/ui/image';
import { Calendar, User, ArrowLeft, Play } from 'lucide-react';
import { format } from 'date-fns';
import { tr } from 'date-fns/locale';

// Video type detection
const getVideoType = (url: string): 'youtube' | 'googledrive' | 'invalid' => {
  if (url.includes('youtube.com') || url.includes('youtu.be')) {
    return 'youtube';
  }
  if (url.includes('drive.google.com')) {
    return 'googledrive';
  }
  return 'invalid';
};

// YouTube video ID extraction function
const getYouTubeVideoId = (url: string): string | null => {
  const regex = /(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/;
  const match = url.match(regex);
  return match ? match[1] : null;
};

// Google Drive video ID extraction function
const getGoogleDriveVideoId = (url: string): string | null => {
  const regex = /\/file\/d\/([a-zA-Z0-9_-]+)/;
  const match = url.match(regex);
  return match ? match[1] : null;
};

// YouTube embed component
const YouTubeEmbed = ({ videoUrl }: { videoUrl: string }) => {
  const videoId = getYouTubeVideoId(videoUrl);
  
  if (!videoId) {
    return (
      <div className="bg-gray-900 rounded-lg p-8 text-center">
        <Play className="w-12 h-12 text-gray-500 mx-auto mb-4" />
        <p className="text-gray-400">Geçersiz YouTube video linki</p>
      </div>
    );
  }

  return (
    <div className="relative w-full" style={{ paddingBottom: '56.25%' }}>
      <iframe
        className="absolute top-0 left-0 w-full h-full rounded-lg"
        src={`https://www.youtube.com/embed/${videoId}?rel=0&modestbranding=1`}
        title="YouTube video player"
        frameBorder="0"
        allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share"
        allowFullScreen
      />
    </div>
  );
};

// Google Drive embed component
const GoogleDriveEmbed = ({ videoUrl }: { videoUrl: string }) => {
  const videoId = getGoogleDriveVideoId(videoUrl);
  
  if (!videoId) {
    return (
      <div className="bg-gray-900 rounded-lg p-8 text-center">
        <Play className="w-12 h-12 text-gray-500 mx-auto mb-4" />
        <p className="text-gray-400">Geçersiz Google Drive video linki</p>
      </div>
    );
  }

  return (
    <div className="relative w-full" style={{ paddingBottom: '56.25%' }}>
      <iframe
        className="absolute top-0 left-0 w-full h-full rounded-lg"
        src={`https://drive.google.com/file/d/${videoId}/preview`}
        title="Google Drive video player"
        frameBorder="0"
        allow="autoplay"
        allowFullScreen
      />
    </div>
  );
};

// Universal video embed component
const VideoEmbed = ({ videoUrl }: { videoUrl: string }) => {
  const videoType = getVideoType(videoUrl);
  
  switch (videoType) {
    case 'youtube':
      return <YouTubeEmbed videoUrl={videoUrl} />;
    case 'googledrive':
      return <GoogleDriveEmbed videoUrl={videoUrl} />;
    default:
      return (
        <div className="bg-gray-900 rounded-lg p-8 text-center">
          <Play className="w-12 h-12 text-gray-500 mx-auto mb-4" />
          <p className="text-gray-400">Desteklenmeyen video formatı</p>
          <p className="text-gray-500 text-sm mt-2">YouTube veya Google Drive linki kullanın</p>
        </div>
      );
  }
};

export default function NewsDetailPage() {
  const { id } = useParams<{ id: string }>();
  const [newsItem, setNewsItem] = useState<Haberler | null>(null);
  const [loading, setLoading] = useState(true);
  const [notFound, setNotFound] = useState(false);

  useEffect(() => {
    const fetchNewsItem = async () => {
      if (!id) {
        setNotFound(true);
        setLoading(false);
        return;
      }

      try {
        const item = await BaseCrudService.getById<Haberler>('haberler', id);
        if (item) {
          setNewsItem(item);
        } else {
          setNotFound(true);
        }
      } catch (error) {
        console.error('Haber yüklenirken hata:', error);
        setNotFound(true);
      } finally {
        setLoading(false);
      }
    };

    fetchNewsItem();
  }, [id]);

  if (loading) {
    return (
      <div className="min-h-screen bg-black flex items-center justify-center">
        <LoadingSpinner />
      </div>
    );
  }

  if (notFound || !newsItem) {
    return (
      <div className="min-h-screen bg-black text-white flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-4xl font-heading font-bold mb-4">Haber Bulunamadı</h1>
          <p className="text-white/60 mb-8">Aradığınız haber mevcut değil veya kaldırılmış olabilir.</p>
          <Link to="/news">
            <Button variant="outline" className="bg-transparent border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-black">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Haberlere Dön
            </Button>
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-black text-white">
      {/* Header */}
      <div className="bg-gradient-to-r from-secondary to-accentneongreen">
        <div className="max-w-[120rem] mx-auto px-6 py-8">
          <div className="flex items-center gap-4 mb-6">
            <Link to="/news">
              <Button variant="outline" size="sm" className="bg-white/10 border-white/20 text-white hover:bg-white/20">
                <ArrowLeft className="w-4 h-4 mr-2" />
                Haberlere Dön
              </Button>
            </Link>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="max-w-4xl mx-auto px-6 py-16">
        {/* Article Header */}
        <div className="mb-8">
          <h1 className="text-4xl md:text-5xl font-heading font-bold mb-6 leading-tight">
            {newsItem.title}
          </h1>
          
          <div className="flex flex-wrap items-center gap-6 text-white/60 mb-8">
            {newsItem.publishDate && (
              <div className="flex items-center gap-2">
                <Calendar className="w-5 h-5" />
                <span>{format(new Date(newsItem.publishDate), 'dd MMMM yyyy, HH:mm', { locale: tr })}</span>
              </div>
            )}
            {newsItem.author && (
              <div className="flex items-center gap-2">
                <User className="w-5 h-5" />
                <span>{newsItem.author}</span>
              </div>
            )}
          </div>

          {newsItem.excerpt && (
            <p className="text-xl text-white/80 leading-relaxed mb-8 font-paragraph">
              {newsItem.excerpt}
            </p>
          )}
        </div>

        {/* Main Image */}
        {newsItem.mainImage && (
          <div className="mb-12 rounded-lg overflow-hidden">
            <Image
              src={newsItem.mainImage}
              alt={newsItem.title || 'Haber görseli'}
              width={800}
              className="w-full h-auto object-cover"
            />
          </div>
        )}

        {/* Video Player */}
        {newsItem.videoUrl && (
          <div className="mb-12">
            <h3 className="text-xl font-heading font-semibold text-white mb-4 flex items-center">
              <Play className="w-5 h-5 mr-2 text-accentneongreen" />
              Video
            </h3>
            <VideoEmbed videoUrl={newsItem.videoUrl} />
          </div>
        )}

        {/* Article Content */}
        <div className="prose prose-invert prose-lg max-w-none">
          <div className="text-white/90 leading-relaxed font-paragraph whitespace-pre-wrap">
            {newsItem.description}
          </div>
        </div>

        {/* Back to News */}
        <div className="mt-16 pt-8 border-t border-gray-800">
          <Link to="/news">
            <Button 
              variant="outline" 
              className="bg-transparent border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-black"
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Tüm Haberleri Görüntüle
            </Button>
          </Link>
        </div>
      </div>
    </div>
  );
}