import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { BaseCrudService } from '@/integrations';
import { Haberler } from '@/entities';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { LoadingSpinner } from '@/components/ui/loading-spinner';
import { Image } from '@/components/ui/image';
import { Calendar, User, ArrowLeft, FileVideo } from 'lucide-react';
import { format } from 'date-fns';
import { tr } from 'date-fns/locale';

// Video type detection
const getVideoType = (url: string): 'youtube' | 'googledrive' | 'invalid' => {
  if (!url) return 'invalid';
  if (url.includes('youtube.com') || url.includes('youtu.be')) {
    return 'youtube';
  }
  if (url.includes('drive.google.com')) {
    return 'googledrive';
  }
  return 'invalid';
};

// Video platform badge component
const VideoPlatformBadge = ({ videoUrl }: { videoUrl: string }) => {
  const videoType = getVideoType(videoUrl);
  
  switch (videoType) {
    case 'youtube':
      return (
        <Badge className="bg-red-500/20 text-red-400 border-red-500/30">
          <FileVideo className="w-3 h-3 mr-1" />
          YouTube
        </Badge>
      );
    case 'googledrive':
      return (
        <Badge className="bg-blue-500/20 text-blue-400 border-blue-500/30">
          <FileVideo className="w-3 h-3 mr-1" />
          Google Drive
        </Badge>
      );
    default:
      return null;
  }
};

export default function NewsPage() {
  const [news, setNews] = useState<Haberler[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchNews = async () => {
      try {
        const { items } = await BaseCrudService.getAll<Haberler>('haberler');
        // Sort by publish date, newest first
        const sortedNews = items.sort((a, b) => {
          const dateA = new Date(a.publishDate || a._createdDate || 0);
          const dateB = new Date(b.publishDate || b._createdDate || 0);
          return dateB.getTime() - dateA.getTime();
        });
        setNews(sortedNews);
      } catch (error) {
        console.error('Haberler yüklenirken hata:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchNews();
  }, []);

  if (loading) {
    return (
      <div className="min-h-screen bg-black flex items-center justify-center">
        <LoadingSpinner />
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-black text-white">
      {/* Header */}
      <div className="bg-gradient-to-r from-secondary to-accentneongreen">
        <div className="max-w-[120rem] mx-auto px-6 py-16">
          <div className="flex items-center gap-4 mb-6">
            <Link to="/">
              <Button variant="outline" size="sm" className="bg-white/10 border-white/20 text-white hover:bg-white/20">
                <ArrowLeft className="w-4 h-4 mr-2" />
                Ana Sayfa
              </Button>
            </Link>
          </div>
          <h1 className="text-5xl font-heading font-bold mb-4">Haberler</h1>
          <p className="text-xl text-white/80 max-w-2xl">
            Anime ve manga dünyasından en güncel haberler, duyurular ve gelişmeler
          </p>
        </div>
      </div>

      {/* News Grid */}
      <div className="max-w-[120rem] mx-auto px-6 py-16">
        {news.length === 0 ? (
          <div className="text-center py-16">
            <h2 className="text-2xl font-heading font-semibold mb-4 text-white/60">
              Henüz haber bulunmuyor
            </h2>
            <p className="text-white/40">
              Yakında anime ve manga dünyasından haberler burada yer alacak.
            </p>
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {news.map((item) => (
              <Card key={item._id} className="bg-gray-900 border-gray-800 hover:border-accentneongreen/50 transition-all duration-300 group overflow-hidden">
                {item.mainImage && (
                  <div className="relative overflow-hidden">
                    <Image
                      src={item.mainImage}
                      alt={item.title || 'Haber görseli'}
                      width={400}
                      className="w-full h-48 object-cover group-hover:scale-105 transition-transform duration-300"
                    />
                    <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent" />
                  </div>
                )}
                
                <CardHeader className="pb-3">
                  <div className="flex items-center gap-4 text-sm text-white/60 mb-3">
                    {item.publishDate && (
                      <div className="flex items-center gap-1">
                        <Calendar className="w-4 h-4" />
                        {format(new Date(item.publishDate), 'dd MMMM yyyy', { locale: tr })}
                      </div>
                    )}
                    {item.author && (
                      <div className="flex items-center gap-1">
                        <User className="w-4 h-4" />
                        {item.author}
                      </div>
                    )}
                    {item.videoUrl && (
                      <VideoPlatformBadge videoUrl={item.videoUrl} />
                    )}
                  </div>
                  
                  <CardTitle className="text-xl font-heading text-white group-hover:text-accentneongreen transition-colors line-clamp-2">
                    {item.title}
                  </CardTitle>
                </CardHeader>
                
                <CardContent>
                  {item.excerpt && (
                    <p className="text-white/70 text-sm line-clamp-3 mb-4">
                      {item.excerpt}
                    </p>
                  )}
                  
                  <Link to={`/news/${item._id}`}>
                    <Button 
                      variant="outline" 
                      size="sm" 
                      className="w-full bg-transparent border-accentneongreen/30 text-accentneongreen hover:bg-accentneongreen hover:text-black transition-all"
                    >
                      Devamını Oku
                    </Button>
                  </Link>
                </CardContent>
              </Card>
            ))}
          </div>
        )}
      </div>
    </div>
  );
}