import { useMember } from '@/integrations';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Image } from '@/components/ui/image';
import { Link } from 'react-router-dom';
import { BookOpen, User, Mail, Calendar, Settings, LogOut } from 'lucide-react';
import { format } from 'date-fns';

export default function ProfilePage() {
  const { member, actions } = useMember();

  const handleLogout = () => {
    actions.logout();
  };

  return (
    <div className="min-h-screen bg-primary">
      {/* Navigation */}
      <nav className="w-full px-6 py-4 flex items-center justify-between max-w-[120rem] mx-auto border-b border-primary-foreground/10">
        <Link to="/" className="flex items-center space-x-2">
          <div className="w-8 h-8 bg-accentneongreen rounded-lg flex items-center justify-center">
            <BookOpen className="w-5 h-5 text-primary" />
          </div>
          <span className="font-heading text-xl font-bold text-primary-foreground">Mangius</span>
        </Link>
        
        <div className="hidden md:flex items-center space-x-8">
          <Link to="/anime" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Anime
          </Link>
          <Link to="/manga" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Manga
          </Link>
          <Link to="/friends" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Arkadaşlar
          </Link>
          <Link to="/" className="font-paragraph text-primary-foreground hover:text-accentneongreen transition-colors">
            Ana Sayfa
          </Link>
        </div>

        <div className="flex items-center space-x-4">
          <Link to="/profile">
            <Button variant="outline" className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary">
              Profil
            </Button>
          </Link>
          <Button 
            onClick={handleLogout}
            variant="outline" 
            className="border-primary-foreground text-primary-foreground hover:bg-primary-foreground hover:text-primary"
          >
            <LogOut className="w-4 h-4 mr-2" />
            Çıkış Yap
          </Button>
        </div>
      </nav>

      {/* Profile Content */}
      <div className="w-full max-w-[120rem] mx-auto px-6 py-16">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Profile Card */}
          <div className="lg:col-span-1">
            <Card className="bg-secondary border-0">
              <CardContent className="p-8 text-center">
                <div className="mb-6">
                  {member?.profile?.photo?.url ? (
                    <Image
                      src={member.profile.photo.url}
                      alt="Profil fotoğrafı"
                      className="w-24 h-24 rounded-full mx-auto object-cover"
                      width={96}
                    />
                  ) : (
                    <div className="w-24 h-24 bg-accentneongreen rounded-full mx-auto flex items-center justify-center">
                      <User className="w-12 h-12 text-primary" />
                    </div>
                  )}
                </div>
                
                <h1 className="font-heading text-2xl font-bold text-secondary-foreground mb-2">
                  {member?.profile?.nickname || member?.contact?.firstName || 'Kullanıcı'}
                </h1>
                
                {member?.profile?.title && (
                  <p className="font-paragraph text-secondary-foreground/70 mb-4">
                    {member.profile.title}
                  </p>
                )}
                
                <Badge className="bg-accentneongreen text-primary mb-6">
                  {member?.status === 'APPROVED' ? 'Aktif Üye' : 'Üye'}
                </Badge>
                
                <div className="space-y-4">
                  <Button className="w-full bg-accentneongreen text-primary hover:bg-accentneongreen/90">
                    <Settings className="w-4 h-4 mr-2" />
                    Profili Düzenle
                  </Button>
                  
                  <Button 
                    variant="outline" 
                    className="w-full border-secondary-foreground text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
                    onClick={handleLogout}
                  >
                    <LogOut className="w-4 h-4 mr-2" />
                    Çıkış Yap
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Profile Details */}
          <div className="lg:col-span-2 space-y-8">
            {/* Account Information */}
            <Card className="bg-secondary border-0">
              <CardHeader>
                <CardTitle className="font-heading text-xl font-bold text-secondary-foreground">
                  Hesap Bilgileri
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="space-y-2">
                    <label className="font-paragraph text-sm font-medium text-secondary-foreground/70">
                      E-posta Adresi
                    </label>
                    <div className="flex items-center space-x-2">
                      <Mail className="w-4 h-4 text-accentneongreen" />
                      <span className="font-paragraph text-secondary-foreground">
                        {member?.loginEmail || 'Belirtilmemiş'}
                      </span>
                      {member?.loginEmailVerified && (
                        <Badge variant="secondary" className="bg-accentneongreen/20 text-accentneongreen text-xs">
                          Doğrulandı
                        </Badge>
                      )}
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="font-paragraph text-sm font-medium text-secondary-foreground/70">
                      Kullanıcı Adı
                    </label>
                    <div className="flex items-center space-x-2">
                      <User className="w-4 h-4 text-accentneongreen" />
                      <span className="font-paragraph text-secondary-foreground">
                        {member?.profile?.nickname || 'Belirtilmemiş'}
                      </span>
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="font-paragraph text-sm font-medium text-secondary-foreground/70">
                      Ad
                    </label>
                    <span className="font-paragraph text-secondary-foreground">
                      {member?.contact?.firstName || 'Belirtilmemiş'}
                    </span>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="font-paragraph text-sm font-medium text-secondary-foreground/70">
                      Soyad
                    </label>
                    <span className="font-paragraph text-secondary-foreground">
                      {member?.contact?.lastName || 'Belirtilmemiş'}
                    </span>
                  </div>
                  
                  {member?.contact?.phones && member.contact.phones.length > 0 && (
                    <div className="space-y-2">
                      <label className="font-paragraph text-sm font-medium text-secondary-foreground/70">
                        Telefon
                      </label>
                      <span className="font-paragraph text-secondary-foreground">
                        {member.contact.phones[0]}
                      </span>
                    </div>
                  )}
                  
                  <div className="space-y-2">
                    <label className="font-paragraph text-sm font-medium text-secondary-foreground/70">
                      Hesap Durumu
                    </label>
                    <Badge 
                      className={
                        member?.status === 'APPROVED' 
                          ? 'bg-accentneongreen text-primary' 
                          : 'bg-secondary-foreground/20 text-secondary-foreground'
                      }
                    >
                      {member?.status === 'APPROVED' ? 'Onaylandı' : 
                       member?.status === 'PENDING' ? 'Beklemede' :
                       member?.status === 'BLOCKED' ? 'Engellenmiş' : 'Bilinmiyor'}
                    </Badge>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Account Activity */}
            <Card className="bg-secondary border-0">
              <CardHeader>
                <CardTitle className="font-heading text-xl font-bold text-secondary-foreground">
                  Hesap Aktivitesi
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="space-y-2">
                    <label className="font-paragraph text-sm font-medium text-secondary-foreground/70">
                      Hesap Oluşturma Tarihi
                    </label>
                    <div className="flex items-center space-x-2">
                      <Calendar className="w-4 h-4 text-accentneongreen" />
                      <span className="font-paragraph text-secondary-foreground">
                        {member?._createdDate 
                          ? format(new Date(member._createdDate), 'dd MMMM yyyy')
                          : 'Bilinmiyor'
                        }
                      </span>
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="font-paragraph text-sm font-medium text-secondary-foreground/70">
                      Son Giriş
                    </label>
                    <div className="flex items-center space-x-2">
                      <Calendar className="w-4 h-4 text-accentneongreen" />
                      <span className="font-paragraph text-secondary-foreground">
                        {member?.lastLoginDate 
                          ? format(new Date(member.lastLoginDate), 'dd MMMM yyyy')
                          : 'Bilinmiyor'
                        }
                      </span>
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="font-paragraph text-sm font-medium text-secondary-foreground/70">
                      Son Güncelleme
                    </label>
                    <div className="flex items-center space-x-2">
                      <Calendar className="w-4 h-4 text-accentneongreen" />
                      <span className="font-paragraph text-secondary-foreground">
                        {member?._updatedDate 
                          ? format(new Date(member._updatedDate), 'dd MMMM yyyy')
                          : 'Bilinmiyor'
                        }
                      </span>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Quick Actions */}
            <Card className="bg-secondary border-0">
              <CardHeader>
                <CardTitle className="font-heading text-xl font-bold text-secondary-foreground">
                  Hızlı İşlemler
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <Link to="/anime">
                    <Button variant="outline" className="w-full border-secondary-foreground text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary">
                      Anime Koleksiyonunu Gözat
                    </Button>
                  </Link>
                  
                  <Link to="/manga">
                    <Button variant="outline" className="w-full border-secondary-foreground text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary">
                      Manga Kütüphanesini Gözat
                    </Button>
                  </Link>
                  
                  <Button variant="outline" className="w-full border-secondary-foreground text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary">
                    İzleme Listesi
                  </Button>
                  
                  <Button variant="outline" className="w-full border-secondary-foreground text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary">
                    Favoriler
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>

      {/* Footer */}
      <footer className="w-full bg-primary border-t border-primary-foreground/10">
        <div className="max-w-[120rem] mx-auto px-6 py-12">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="space-y-4">
              <div className="flex items-center space-x-2">
                <div className="w-8 h-8 bg-accentneongreen rounded-lg flex items-center justify-center">
                  <BookOpen className="w-5 h-5 text-primary" />
                </div>
                <span className="font-heading text-xl font-bold text-primary-foreground">Mangius</span>
              </div>
              <p className="font-paragraph text-primary-foreground/70">
                En iyi anime ve manga içeriklerine açılan kapınız.
              </p>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Gözat</h3>
              <div className="space-y-2">
                <Link to="/anime" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Anime Koleksiyonu
                </Link>
                <Link to="/manga" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Manga Kütüphanesi
                </Link>
                <Link to="/" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Ana Sayfa
                </Link>
              </div>
            </div>
            
            <div className="space-y-4">
              <h3 className="font-heading text-lg font-bold text-primary-foreground">Destek</h3>
              <div className="space-y-2">
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  Yardım Merkezi
                </a>
                <a href="#" className="block font-paragraph text-primary-foreground/70 hover:text-accentneongreen transition-colors">
                  İletişim
                </a>
              </div>
            </div>
          </div>
          
          <div className="border-t border-primary-foreground/10 mt-8 pt-8 text-center">
            <p className="font-paragraph text-primary-foreground/60">
              © 2024 Mangius. Tüm hakları saklıdır.
            </p>
          </div>
        </div>
      </footer>
    </div>
  );
}