import { useMember } from '@/integrations';
import { LoadingSpinner } from '@/components/ui/loading-spinner';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Shield, AlertTriangle, Home, LogOut } from 'lucide-react';
import { Link } from 'react-router-dom';

interface AdminProtectedRouteProps {
  children: React.ReactNode;
  allowedEmail?: string;
}

export function AdminProtectedRoute({ 
  children, 
  allowedEmail = 'berkcanguler3@gmail.com' 
}: AdminProtectedRouteProps) {
  const { member, isAuthenticated, isLoading, actions } = useMember();

  // Show loading spinner while checking authentication
  if (isLoading) {
    return (
      <div className="min-h-screen bg-primary flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner />
          <p className="font-paragraph text-primary-foreground mt-4">
            Yetki kontrolü yapılıyor...
          </p>
        </div>
      </div>
    );
  }

  // If not authenticated, show login prompt
  if (!isAuthenticated || !member) {
    return (
      <div className="min-h-screen bg-primary flex items-center justify-center p-6">
        <Card className="bg-secondary border-0 max-w-md w-full">
          <CardHeader className="text-center">
            <div className="w-16 h-16 bg-accentneongreen/20 rounded-full flex items-center justify-center mx-auto mb-4">
              <Shield className="w-8 h-8 text-accentneongreen" />
            </div>
            <CardTitle className="font-heading text-2xl font-bold text-secondary-foreground">
              Admin Paneli Erişimi
            </CardTitle>
          </CardHeader>
          <CardContent className="text-center space-y-6">
            <p className="font-paragraph text-secondary-foreground/70">
              Admin paneline erişmek için giriş yapmanız gerekiyor.
            </p>
            <div className="space-y-3">
              <Button 
                onClick={actions.login}
                className="w-full bg-accentneongreen text-primary hover:bg-accentneongreen/90"
              >
                Giriş Yap
              </Button>
              <Link to="/">
                <Button variant="outline" className="w-full border-secondary-foreground/30 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary">
                  <Home className="w-4 h-4 mr-2" />
                  Ana Sayfaya Dön
                </Button>
              </Link>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  // Check if user has admin privileges (specific email)
  const userEmail = member.loginEmail;
  const isAdmin = userEmail === allowedEmail;

  if (!isAdmin) {
    return (
      <div className="min-h-screen bg-primary flex items-center justify-center p-6">
        <Card className="bg-secondary border-0 max-w-md w-full">
          <CardHeader className="text-center">
            <div className="w-16 h-16 bg-red-500/20 rounded-full flex items-center justify-center mx-auto mb-4">
              <AlertTriangle className="w-8 h-8 text-red-500" />
            </div>
            <CardTitle className="font-heading text-2xl font-bold text-secondary-foreground">
              Erişim Reddedildi
            </CardTitle>
          </CardHeader>
          <CardContent className="text-center space-y-6">
            <div className="space-y-3">
              <p className="font-paragraph text-secondary-foreground/70">
                Bu sayfaya erişim yetkiniz bulunmamaktadır.
              </p>
              <p className="font-paragraph text-secondary-foreground/60 text-sm">
                Admin paneline sadece yetkili kullanıcılar erişebilir.
              </p>
              {userEmail && (
                <div className="bg-primary/20 rounded-lg p-3">
                  <p className="font-paragraph text-secondary-foreground/60 text-xs">
                    Mevcut hesap: {userEmail}
                  </p>
                </div>
              )}
            </div>
            <div className="space-y-3">
              <Link to="/">
                <Button className="w-full bg-accentneongreen text-primary hover:bg-accentneongreen/90">
                  <Home className="w-4 h-4 mr-2" />
                  Ana Sayfaya Dön
                </Button>
              </Link>
              <Button 
                onClick={actions.logout}
                variant="outline" 
                className="w-full border-secondary-foreground/30 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
              >
                <LogOut className="w-4 h-4 mr-2" />
                Farklı Hesapla Giriş Yap
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  // If user is admin, render the protected content
  return <>{children}</>;
}