import { useState, useRef } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Progress } from '@/components/ui/progress';
import { Upload, X, CheckCircle, AlertCircle, FileVideo, Play, Server, Link } from 'lucide-react';
import { BaseCrudService } from '@/integrations';
import { AnimeEpisodes } from '@/entities';
import { ServerVideoStorage } from '@/lib/server-video-storage';

interface AnimeEpisodeUploadDialogProps {
  isOpen: boolean;
  onClose: () => void;
  animeId: string;
  animeTitle?: string;
  onEpisodeAdded?: () => void;
}

export function AnimeEpisodeUploadDialog({ 
  isOpen, 
  onClose, 
  animeId, 
  animeTitle,
  onEpisodeAdded 
}: AnimeEpisodeUploadDialogProps) {
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [megaUrl, setMegaUrl] = useState('');
  const [uploadType, setUploadType] = useState<'file' | 'mega'>('file');
  const [episodeNumber, setEpisodeNumber] = useState('');
  const [episodeTitle, setEpisodeTitle] = useState('');
  const [episodeDescription, setEpisodeDescription] = useState('');
  const [duration, setDuration] = useState('');
  const [isUploading, setIsUploading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [uploadStatus, setUploadStatus] = useState<'idle' | 'uploading' | 'success' | 'error'>('idle');
  const [errorMessage, setErrorMessage] = useState('');
  const fileInputRef = useRef<HTMLInputElement>(null);

  const validateMegaUrl = (url: string) => {
    const megaPatterns = [
      /^https?:\/\/mega\.nz\/file\/[A-Za-z0-9_-]+#[A-Za-z0-9_-]+$/,
      /^https?:\/\/mega\.co\.nz\/file\/[A-Za-z0-9_-]+#[A-Za-z0-9_-]+$/,
      /^https?:\/\/mega\.nz\/#![A-Za-z0-9_-]+![A-Za-z0-9_-]+$/,
      /^https?:\/\/mega\.co\.nz\/#![A-Za-z0-9_-]+![A-Za-z0-9_-]+$/
    ];
    
    return megaPatterns.some(pattern => pattern.test(url));
  };

  const handleFileSelect = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      // Desteklenen video formatları
      const supportedFormats = [
        'video/mp4',
        'video/webm', 
        'video/ogg',
        'video/avi',
        'video/mov',
        'video/quicktime'
      ];
      
      // Dosya uzantısı kontrolü
      const fileName = file.name.toLowerCase();
      const supportedExtensions = ['.mp4', '.webm', '.ogg', '.avi', '.mov'];
      const hasValidExtension = supportedExtensions.some(ext => fileName.endsWith(ext));
      
      // MIME type ve uzantı kontrolü
      if (!file.type.startsWith('video/') && !hasValidExtension) {
        setErrorMessage('Desteklenmeyen dosya formatı. Lütfen MP4, WebM, OGG, AVI veya MOV formatında ve maksimum 2GB boyutunda bir video dosyası seçin.');
        return;
      }
      
      // Özel format kontrolü
      if (!supportedFormats.includes(file.type) && !hasValidExtension) {
        setErrorMessage('Bu video formatı desteklenmiyor. Desteklenen formatlar: MP4, WebM, OGG, AVI, MOV (Maksimum: 2GB)');
        return;
      }
      
      // Dosya boyutu kontrolü (max 2GB for episodes)
      const maxSize = 2 * 1024 * 1024 * 1024; // 2GB
      if (file.size > maxSize) {
        setErrorMessage('Video dosyası 2GB\'dan küçük olmalıdır.');
        return;
      }

      // Minimum dosya boyutu kontrolü (1KB)
      if (file.size < 1024) {
        setErrorMessage('Dosya çok küçük. Lütfen geçerli bir video dosyası seçin (maksimum 2GB).');
        return;
      }

      setSelectedFile(file);
      setErrorMessage('');
    }
  };

  const simulateUpload = async () => {
    if (uploadType === 'file' && !selectedFile) {
      setErrorMessage('Lütfen önce bir video dosyası seçin (maksimum 2GB).');
      return;
    }
    
    if (uploadType === 'mega' && !megaUrl.trim()) {
      setErrorMessage('Lütfen geçerli bir Mega.nz linki girin.');
      return;
    }
    
    if (uploadType === 'mega' && !validateMegaUrl(megaUrl.trim())) {
      setErrorMessage('Geçersiz Mega.nz linki. Lütfen doğru formatta bir link girin.');
      return;
    }

    setIsUploading(true);
    setUploadStatus('uploading');
    setUploadProgress(0);
    setErrorMessage('');

    try {
      let videoUrl = '';
      let thumbnailUrl = '';

      if (uploadType === 'mega') {
        // Mega.nz URL için simülasyon
        const totalSteps = 50; // Mega için daha hızlı
        const stepDelay = 60;
        
        for (let i = 0; i <= totalSteps; i += 2) {
          await new Promise(resolve => setTimeout(resolve, stepDelay));
          setUploadProgress(i);
        }
        
        videoUrl = megaUrl.trim();
        thumbnailUrl = 'https://static.wixstatic.com/media/a92b31_fba2171573334ae2bd7a8ed0af41d32a~mv2.png?originWidth=256&originHeight=192';
      } else {
        // Dosya yükleme simülasyonu
        const totalSteps = 100;
        const stepDelay = 120;
        
        for (let i = 0; i <= totalSteps; i += 1) {
          await new Promise(resolve => setTimeout(resolve, stepDelay));
          setUploadProgress(i);
          
          if (i === 20 || i === 40 || i === 60 || i === 80) {
            await new Promise(resolve => setTimeout(resolve, 400));
          }
        }

        if (!selectedFile!.type.startsWith('video/')) {
          throw new Error('Seçilen dosya geçerli bir video dosyası değil.');
        }

        const { videoId, videoUrl: serverVideoUrl, thumbnailUrl: serverThumbnailUrl } = await ServerVideoStorage.uploadVideo(selectedFile!, {
          title: episodeTitle,
          description: episodeDescription,
          category: 'anime-episode',
          animeId: animeId,
          episodeNumber: episodeNumber ? parseInt(episodeNumber) : undefined
        });
        
        videoUrl = serverVideoUrl;
        thumbnailUrl = serverThumbnailUrl;
      }
      
      // Create episode entry in database
      const newEpisode = {
        _id: crypto.randomUUID(),
        animeId: animeId,
        episodeNumber: episodeNumber ? parseInt(episodeNumber) : undefined,
        episodeTitle: episodeTitle,
        episodeDescription: episodeDescription,
        videoUrl: videoUrl,
        thumbnail: thumbnailUrl,
        duration: duration ? parseInt(duration) : undefined
      };

      await BaseCrudService.create('animeepisodes', newEpisode as any);
      
      setUploadStatus('success');
      setIsUploading(false);
      
      if (onEpisodeAdded) {
        onEpisodeAdded();
      }
      
      setTimeout(() => {
        resetForm();
        onClose();
      }, 2000);
      
    } catch (error) {
      console.error('Upload error:', error);
      setUploadStatus('error');
      setIsUploading(false);
      
      if (error instanceof Error) {
        setErrorMessage(`Yükleme hatası: ${error.message}. Video dosyanızın 2GB'dan küçük olduğundan emin olun.`);
      } else {
        setErrorMessage('Video yüklenirken beklenmeyen bir hata oluştu. Lütfen dosyanızın 2GB\'dan küçük ve desteklenen formatta (MP4, WebM, OGG, AVI, MOV) olduğundan emin olun ve tekrar deneyin.');
      }
    }
  };

  const resetForm = () => {
    setSelectedFile(null);
    setMegaUrl('');
    setUploadType('file');
    setEpisodeNumber('');
    setEpisodeTitle('');
    setEpisodeDescription('');
    setDuration('');
    setUploadProgress(0);
    setUploadStatus('idle');
    setErrorMessage('');
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const handleClose = () => {
    if (!isUploading) {
      resetForm();
      onClose();
    }
  };

  const canSubmit = (
    (uploadType === 'file' && selectedFile) || 
    (uploadType === 'mega' && megaUrl.trim() && validateMegaUrl(megaUrl.trim()))
  ) && episodeNumber.trim() && episodeTitle.trim() && !isUploading;

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-[700px] bg-secondary border-0 max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="font-heading text-2xl font-bold text-secondary-foreground flex items-center gap-2">
            <Play className="w-6 h-6 text-accentneongreen" />
            Anime Bölümü Yükle
          </DialogTitle>
          {animeTitle && (
            <p className="font-paragraph text-secondary-foreground/70 text-sm mt-1">
              {animeTitle} için yeni bölüm ekleniyor
            </p>
          )}
        </DialogHeader>

        <div className="space-y-6 py-4">
          {/* Upload Type Selection */}
          <div className="space-y-3">
            <Label className="font-paragraph text-secondary-foreground font-medium">Video Kaynağı</Label>
            <div className="flex gap-4">
              <button
                type="button"
                onClick={() => setUploadType('file')}
                className={`flex-1 p-4 rounded-lg border-2 transition-colors ${
                  uploadType === 'file'
                    ? 'border-accentneongreen bg-accentneongreen/10 text-secondary-foreground'
                    : 'border-secondary-foreground/30 text-secondary-foreground/60 hover:border-accentneongreen/50'
                }`}
                disabled={isUploading}
              >
                <div className="flex items-center justify-center gap-2">
                  <FileVideo className="w-5 h-5" />
                  <span className="font-paragraph font-medium">Dosya Yükle</span>
                </div>
                <p className="text-xs mt-1 opacity-70">Bilgisayarınızdan video dosyası</p>
              </button>
              
              <button
                type="button"
                onClick={() => setUploadType('mega')}
                className={`flex-1 p-4 rounded-lg border-2 transition-colors ${
                  uploadType === 'mega'
                    ? 'border-accentneongreen bg-accentneongreen/10 text-secondary-foreground'
                    : 'border-secondary-foreground/30 text-secondary-foreground/60 hover:border-accentneongreen/50'
                }`}
                disabled={isUploading}
              >
                <div className="flex items-center justify-center gap-2">
                  <Link className="w-5 h-5" />
                  <span className="font-paragraph font-medium">Mega.nz Link</span>
                </div>
                <p className="text-xs mt-1 opacity-70">Mega.nz video bağlantısı</p>
              </button>
            </div>
          </div>

          {/* File Upload Area */}
          {uploadType === 'file' && (
            <div className="space-y-2">
              <Label className="font-paragraph text-secondary-foreground font-medium">Video Dosyası</Label>
              <div 
                className={`border-2 border-dashed rounded-lg p-6 text-center transition-colors ${
                  selectedFile 
                    ? 'border-accentneongreen bg-accentneongreen/5' 
                    : 'border-secondary-foreground/30 hover:border-accentneongreen/50'
                }`}
              >
                {selectedFile ? (
                  <div className="space-y-2">
                    <FileVideo className="w-10 h-10 text-accentneongreen mx-auto" />
                    <p className="font-paragraph text-secondary-foreground font-medium">
                      {selectedFile.name}
                    </p>
                    <p className="font-paragraph text-secondary-foreground/60 text-sm">
                      {(selectedFile.size / (1024 * 1024)).toFixed(2)} MB
                    </p>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => {
                        setSelectedFile(null);
                        if (fileInputRef.current) fileInputRef.current.value = '';
                      }}
                      className="border-secondary-foreground/30 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
                    >
                      <X className="w-4 h-4 mr-2" />
                      Kaldır
                    </Button>
                  </div>
                ) : (
                  <div className="space-y-2">
                    <Upload className="w-10 h-10 text-secondary-foreground/40 mx-auto" />
                    <p className="font-paragraph text-secondary-foreground">
                      Video dosyanızı buraya sürükleyin veya seçin
                    </p>
                    <p className="font-paragraph text-secondary-foreground/60 text-sm">
                      <strong>Desteklenen formatlar:</strong> MP4, WebM, OGG, AVI, MOV
                      <br />
                      <strong>Maksimum boyut:</strong> 2GB
                      <br />
                      <Server className="w-4 h-4 inline mr-1 text-accentneongreen" />
                      Video dosyası sitenin sunucu sistemine güvenli şekilde yüklenecektir
                    </p>
                    <Button
                      variant="outline"
                      onClick={() => fileInputRef.current?.click()}
                      className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary"
                    >
                      <Upload className="w-4 h-4 mr-2" />
                      Video Dosyası Seç (Max: 2GB)
                    </Button>
                  </div>
                )}
                <input
                  ref={fileInputRef}
                  type="file"
                  accept=".mp4,.webm,.ogg,video/mp4,video/webm,video/ogg"
                  onChange={handleFileSelect}
                  className="hidden"
                />
              </div>
            </div>
          )}

          {/* Mega URL Input */}
          {uploadType === 'mega' && (
            <div className="space-y-2">
              <Label htmlFor="megaUrl" className="font-paragraph text-secondary-foreground font-medium">
                Mega.nz Video Linki
              </Label>
              <Input
                id="megaUrl"
                value={megaUrl}
                onChange={(e) => setMegaUrl(e.target.value)}
                placeholder="https://mega.nz/file/XXXXXXXX#XXXXXXXXXXXXXXXXXXXXXXXX"
                className="bg-primary border-secondary-foreground/30 text-primary-foreground"
                disabled={isUploading}
              />
              <div className="flex items-start gap-2 p-3 bg-accentneongreen/10 border border-accentneongreen/30 rounded-lg">
                <Link className="w-4 h-4 text-accentneongreen mt-0.5 flex-shrink-0" />
                <div className="text-sm">
                  <p className="font-paragraph text-secondary-foreground font-medium mb-1">
                    Desteklenen Mega.nz Link Formatları:
                  </p>
                  <ul className="font-paragraph text-secondary-foreground/70 text-xs space-y-1">
                    <li>• https://mega.nz/file/XXXXXXXX#XXXXXXXXXXXXXXXXXXXXXXXX</li>
                    <li>• https://mega.nz/#!XXXXXXXX!XXXXXXXXXXXXXXXXXXXXXXXX</li>
                    <li>• https://mega.co.nz/file/XXXXXXXX#XXXXXXXXXXXXXXXXXXXXXXXX</li>
                  </ul>
                  <p className="font-paragraph text-secondary-foreground/60 text-xs mt-2">
                    <strong>Not:</strong> Mega.nz linklerinde dosya boyutu sınırı yoktur
                  </p>
                </div>
              </div>
            </div>
          )}

          {/* Form Fields */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="episodeNumber" className="font-paragraph text-secondary-foreground font-medium">
                Bölüm Numarası *
              </Label>
              <Input
                id="episodeNumber"
                type="number"
                value={episodeNumber}
                onChange={(e) => setEpisodeNumber(e.target.value)}
                placeholder="1"
                min="1"
                className="bg-primary border-secondary-foreground/30 text-primary-foreground"
                disabled={isUploading}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="duration" className="font-paragraph text-secondary-foreground font-medium">
                Süre (dakika)
              </Label>
              <Input
                id="duration"
                type="number"
                value={duration}
                onChange={(e) => setDuration(e.target.value)}
                placeholder="24"
                min="1"
                className="bg-primary border-secondary-foreground/30 text-primary-foreground"
                disabled={isUploading}
              />
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="episodeTitle" className="font-paragraph text-secondary-foreground font-medium">
              Bölüm Başlığı *
            </Label>
            <Input
              id="episodeTitle"
              value={episodeTitle}
              onChange={(e) => setEpisodeTitle(e.target.value)}
              placeholder="Örn: İlk Karşılaşma"
              className="bg-primary border-secondary-foreground/30 text-primary-foreground"
              disabled={isUploading}
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="episodeDescription" className="font-paragraph text-secondary-foreground font-medium">
              Bölüm Açıklaması
            </Label>
            <Textarea
              id="episodeDescription"
              value={episodeDescription}
              onChange={(e) => setEpisodeDescription(e.target.value)}
              placeholder="Bu bölümde neler olacağını kısaca açıklayın (isteğe bağlı)..."
              rows={3}
              className="bg-primary border-secondary-foreground/30 text-primary-foreground resize-none"
              disabled={isUploading}
            />
          </div>

          {/* Upload Progress */}
          {uploadStatus === 'uploading' && (
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <Label className="font-paragraph text-secondary-foreground font-medium">
                  Yükleme İlerlemesi
                </Label>
                <span className="font-paragraph text-secondary-foreground/60 text-sm">
                  %{uploadProgress}
                </span>
              </div>
              <Progress value={uploadProgress} className="h-2" />
              <p className="font-paragraph text-secondary-foreground/60 text-sm">
                {uploadType === 'mega' 
                  ? 'Mega.nz linki işleniyor, lütfen bekleyin...'
                  : 'Video dosyası sunucuya yükleniyor, lütfen bekleyin...'
                }
              </p>
            </div>
          )}

          {/* Success Message */}
          {uploadStatus === 'success' && (
            <div className="flex items-center gap-2 p-4 bg-green-500/10 border border-green-500/30 rounded-lg">
              <CheckCircle className="w-5 h-5 text-green-400" />
              <p className="font-paragraph text-green-400">
                {uploadType === 'mega' 
                  ? 'Mega.nz linki başarıyla eklendi! Video artık sitede oynatılabilir.'
                  : 'Video başarıyla sunucuya yüklendi! Herkes artık bu videoyu doğrudan siteden izleyebilir.'
                }
              </p>
            </div>
          )}

          {/* Error Message */}
          {(uploadStatus === 'error' || errorMessage) && (
            <div className="flex items-center gap-2 p-4 bg-red-500/10 border border-red-500/30 rounded-lg">
              <AlertCircle className="w-5 h-5 text-red-400" />
              <p className="font-paragraph text-red-400">
                {errorMessage || 'Bir hata oluştu. Lütfen tekrar deneyin.'}
              </p>
            </div>
          )}

          {/* Action Buttons */}
          <div className="flex justify-end gap-3 pt-4">
            <Button
              variant="outline"
              onClick={handleClose}
              disabled={isUploading}
              className="border-secondary-foreground/30 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
            >
              İptal
            </Button>
            <Button
              onClick={simulateUpload}
              disabled={!canSubmit}
              className="bg-accentneongreen text-primary hover:bg-accentneongreen/90 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isUploading ? (
                <>
                  <div className="w-4 h-4 border-2 border-primary border-t-transparent rounded-full animate-spin mr-2" />
                  Yükleniyor...
                </>
              ) : (
                <>
                  <Upload className="w-4 h-4 mr-2" />
                  {uploadType === 'mega' ? 'Mega Linki Ekle' : 'Sunucuya Yükle'}
                </>
              )}
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}