import { useState, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { BaseCrudService } from '@/integrations';
import { Notifications } from '@/entities';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { X, AlertCircle, ExternalLink } from 'lucide-react';

interface AutoNotificationPopupProps {
  onClose?: () => void;
}

export function AutoNotificationPopup({ onClose }: AutoNotificationPopupProps) {
  const [notification, setNotification] = useState<Notifications | null>(null);
  const [isVisible, setIsVisible] = useState(false);

  useEffect(() => {
    fetchImportantNotification();
  }, []);

  const fetchImportantNotification = async () => {
    try {
      const response = await BaseCrudService.getAll<Notifications>('notifications');
      // Get the most important active notification (lowest display order)
      const importantNotifications = response.items
        .filter(notification => notification.isActive)
        .sort((a, b) => (a.displayOrder || 999) - (b.displayOrder || 999));
      
      if (importantNotifications.length > 0) {
        // Check if user has already seen this notification today
        const notificationId = importantNotifications[0]._id;
        const seenKey = `notification_seen_${notificationId}`;
        const lastSeen = localStorage.getItem(seenKey);
        const today = new Date().toDateString();
        
        if (lastSeen !== today) {
          setNotification(importantNotifications[0]);
          setIsVisible(true);
        }
      }
    } catch (error) {
      console.error('Error fetching important notification:', error);
    }
  };

  const handleClose = () => {
    if (notification) {
      // Mark as seen for today
      const seenKey = `notification_seen_${notification._id}`;
      const today = new Date().toDateString();
      localStorage.setItem(seenKey, today);
    }
    
    setIsVisible(false);
    onClose?.();
  };

  const handleNotificationClick = () => {
    if (notification?.redirectionLink) {
      window.open(notification.redirectionLink, '_blank');
    }
  };

  if (!notification || !isVisible) {
    return null;
  }

  return (
    <AnimatePresence>
      <motion.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        exit={{ opacity: 0 }}
        className="fixed inset-0 bg-black/50 backdrop-blur-sm z-50 flex items-center justify-center p-4"
        onClick={handleClose}
      >
        <motion.div
          initial={{ opacity: 0, scale: 0.9, y: 20 }}
          animate={{ opacity: 1, scale: 1, y: 0 }}
          exit={{ opacity: 0, scale: 0.9, y: 20 }}
          className="w-full max-w-md"
          onClick={(e) => e.stopPropagation()}
        >
          <Card className="bg-gradient-to-br from-primary via-primary/95 to-secondary/20 border border-accentneongreen/30 backdrop-blur-xl shadow-2xl overflow-hidden">
            <CardHeader className="pb-4 border-b border-accentneongreen/20">
              <div className="flex items-center justify-between">
                <CardTitle className="text-xl font-bold text-primary-foreground flex items-center">
                  <div className="p-2 bg-accentneongreen/20 rounded-lg mr-3">
                    <AlertCircle className="w-5 h-5 text-accentneongreen" />
                  </div>
                  Önemli Bildirim
                </CardTitle>
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={handleClose}
                  className="text-primary-foreground/70 hover:text-primary-foreground hover:bg-primary-foreground/10"
                >
                  <X className="w-4 h-4" />
                </Button>
              </div>
            </CardHeader>
            
            <CardContent className="p-6">
              <div className="space-y-4">
                <h3 className="font-heading text-lg font-bold text-primary-foreground">
                  {notification.title}
                </h3>
                
                {notification.content && (
                  <p className="font-paragraph text-primary-foreground/80 leading-relaxed">
                    {notification.content}
                  </p>
                )}
                
                <div className="flex space-x-3 pt-2">
                  {notification.redirectionLink && (
                    <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }} className="flex-1">
                      <Button 
                        onClick={handleNotificationClick}
                        className="w-full bg-gradient-to-r from-accentneongreen to-accentneongreen/80 text-primary hover:from-accentneongreen/90 hover:to-accentneongreen/70 shadow-lg transition-all duration-300"
                      >
                        <ExternalLink className="w-4 h-4 mr-2" />
                        Detayları Gör
                      </Button>
                    </motion.div>
                  )}
                  <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                    <Button 
                      onClick={handleClose}
                      variant="outline"
                      className="border-primary-foreground/30 text-primary-foreground hover:bg-primary-foreground/10 transition-all duration-300"
                    >
                      Kapat
                    </Button>
                  </motion.div>
                </div>
              </div>
            </CardContent>
          </Card>
        </motion.div>
      </motion.div>
    </AnimatePresence>
  );
}