import { useState, useEffect } from 'react';
import { BaseCrudService } from '@/integrations';
import { useMember } from '@/integrations';
import { Yorumlar, KullancProfilleri } from '@/entities';
import type { Member } from '@/types/member';
import { useCommentLikes } from '@/hooks/useCommentLikes';
import { Button } from '@/components/ui/button';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent } from '@/components/ui/card';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';
import { MessageSquare, Send, User, Calendar, Trash2, ThumbsUp, ThumbsDown, MessageCircle } from 'lucide-react';
import { format } from 'date-fns';
import { tr } from 'date-fns/locale';

interface CommentSectionProps {
  contentId: string;
  contentType: 'anime' | 'manga';
}

export function CommentSection({ contentId, contentType }: CommentSectionProps) {
  const { member, isAuthenticated, actions } = useMember();
  const { 
    toggleCommentLike, 
    getCommentLikeStats, 
    getUserLikeForComment, 
    loading: likesLoading 
  } = useCommentLikes();
  
  const [comments, setComments] = useState<Yorumlar[]>([]);
  const [userProfiles, setUserProfiles] = useState<{ [key: string]: KullancProfilleri }>({});
  const [newComment, setNewComment] = useState('');
  const [isLoading, setIsLoading] = useState(true);
  const [isSubmitting, setIsSubmitting] = useState(false);

  useEffect(() => {
    fetchComments();
  }, [contentId, contentType]);

  const fetchComments = async () => {
    setIsLoading(true);
    try {
      const response = await BaseCrudService.getAll<Yorumlar>('yorumlar');
      const contentComments = response.items
        .filter(comment => comment.contentId === contentId && comment.contentType === contentType)
        .sort((a, b) => {
          const dateA = new Date(a.creationDate || a._createdDate || 0);
          const dateB = new Date(b.creationDate || b._createdDate || 0);
          return dateB.getTime() - dateA.getTime(); // En yeni yorumlar üstte
        });
      setComments(contentComments);
      
      // Kullanıcı profillerini yükle
      await loadUserProfiles(contentComments);
    } catch (error) {
      console.error('Yorumlar yüklenirken hata:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const loadUserProfiles = async (comments: Yorumlar[]) => {
    try {
      const { items } = await BaseCrudService.getAll<KullancProfilleri>('kullancprofilleri');
      const profiles: { [key: string]: KullancProfilleri } = {};
      
      comments.forEach(comment => {
        if (comment.userId) {
          const profile = items.find(p => p.memberId === comment.userId);
          if (profile) {
            profiles[comment.userId] = profile;
          }
        }
      });
      
      setUserProfiles(profiles);
    } catch (error) {
      console.error('Kullanıcı profilleri yüklenemedi:', error);
    }
  };

  const handleSubmitComment = async () => {
    if (!isAuthenticated || !member || !newComment.trim()) {
      if (!isAuthenticated) {
        actions.login();
      }
      return;
    }

    setIsSubmitting(true);
    try {
      const commentData: Yorumlar = {
        _id: crypto.randomUUID(),
        commentText: newComment.trim(),
        contentId,
        contentType,
        userId: member.loginEmail || 'anonymous',
        creationDate: new Date().toISOString(),
        _createdDate: new Date(),
        _updatedDate: new Date()
      };

      await BaseCrudService.create('yorumlar', commentData);
      setNewComment('');
      await fetchComments(); // Refresh comments
    } catch (error) {
      console.error('Yorum gönderilirken hata:', error);
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleDeleteComment = async (commentId: string, commentUserId: string) => {
    if (!isAuthenticated || !member) return;
    
    // Only allow users to delete their own comments
    const currentUserId = member.loginEmail || 'anonymous';
    if (commentUserId !== currentUserId) return;

    try {
      await BaseCrudService.delete('yorumlar', commentId);
      await fetchComments(); // Refresh comments
    } catch (error) {
      console.error('Yorum silinirken hata:', error);
    }
  };

  const handleLikeToggle = async (commentId: string, likeType: 'like' | 'dislike') => {
    if (!isAuthenticated) {
      actions.login();
      return;
    }
    
    await toggleCommentLike(commentId, likeType);
  };

  const getUserDisplayName = (userId: string) => {
    // Önce profil verilerinden kontrol et
    const userProfile = userProfiles[userId];
    if (userProfile?.displayName) {
      return userProfile.displayName;
    }
    
    // If it's the current user, show their nickname
    if (member && (member.loginEmail === userId)) {
      return member.profile?.nickname || member.contact?.firstName || 'Kullanıcı';
    }
    // For other users, show a generic name
    return 'Kullanıcı';
  };

  const getUserAvatar = (userId: string) => {
    const userProfile = userProfiles[userId];
    return userProfile?.profilePicture;
  };

  if (isLoading) {
    return (
      <div className="space-y-4">
        <div className="flex items-center space-x-2">
          <MessageSquare className="w-5 h-5 text-accentneongreen" />
          <h3 className="font-heading text-lg font-bold text-secondary-foreground">
            Yorumlar Yükleniyor...
          </h3>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center space-x-2">
          <MessageSquare className="w-5 h-5 text-accentneongreen" />
          <h3 className="font-heading text-lg font-bold text-secondary-foreground">
            Yorumlar
          </h3>
          <Badge className="bg-accentneongreen text-primary">
            {comments.length}
          </Badge>
        </div>
      </div>

      {/* Comment Form */}
      {isAuthenticated ? (
        <Card className="bg-secondary border-0">
          <CardContent className="p-4">
            <div className="space-y-4">
              <div className="flex items-center space-x-3">
                <Avatar className="w-8 h-8">
                  <AvatarImage src={member?.profile?.photo?.url} />
                  <AvatarFallback className="bg-accentneongreen text-primary text-sm">
                    {(member?.profile?.nickname || member?.contact?.firstName || 'K').charAt(0)}
                  </AvatarFallback>
                </Avatar>
                <span className="text-sm font-medium text-secondary-foreground">
                  {member?.profile?.nickname || member?.contact?.firstName || 'Kullanıcı'}
                </span>
              </div>
              <Textarea
                placeholder="Yorumunuzu yazın..."
                value={newComment}
                onChange={(e) => setNewComment(e.target.value)}
                className="bg-primary border-primary-foreground/20 text-primary-foreground placeholder:text-primary-foreground/60"
                rows={3}
              />
              <div className="flex justify-end">
                <Button
                  onClick={handleSubmitComment}
                  disabled={!newComment.trim() || isSubmitting}
                  className="bg-accentneongreen text-primary hover:bg-accentneongreen/90"
                >
                  <Send className="w-4 h-4 mr-2" />
                  {isSubmitting ? 'Gönderiliyor...' : 'Yorum Gönder'}
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      ) : (
        <Card className="bg-secondary border-0">
          <CardContent className="p-6 text-center">
            <MessageCircle className="w-12 h-12 mx-auto mb-4 text-secondary-foreground/60" />
            <h4 className="text-lg font-semibold text-secondary-foreground mb-2">
              Yorum yapmak için giriş yapın
            </h4>
            <p className="text-secondary-foreground/70 mb-4">
              Düşüncelerinizi paylaşmak için hesabınıza giriş yapmanız gerekiyor.
            </p>
            <Button
              onClick={actions.login}
              className="bg-accentneongreen text-primary hover:bg-accentneongreen/90"
            >
              Giriş Yap
            </Button>
          </CardContent>
        </Card>
      )}

      {/* Comments List */}
      <div className="space-y-4">
        {comments.length === 0 ? (
          <Card className="bg-secondary border-0">
            <CardContent className="p-8 text-center">
              <MessageSquare className="w-12 h-12 text-secondary-foreground/30 mx-auto mb-4" />
              <h4 className="font-heading text-lg font-bold text-secondary-foreground mb-2">
                Henüz yorum yok
              </h4>
              <p className="text-secondary-foreground/70">
                İlk yorumu yapan siz olun!
              </p>
            </CardContent>
          </Card>
        ) : (
          comments.map((comment) => {
            const isOwnComment = member && (member.loginEmail === comment.userId);
            const { likeCount, dislikeCount } = getCommentLikeStats(comment._id);
            const userLike = getUserLikeForComment(comment._id);

            return (
              <Card key={comment._id} className="bg-secondary border-0">
                <CardContent className="p-4">
                  <div className="flex space-x-4">
                    <Avatar className="w-10 h-10 flex-shrink-0">
                      <AvatarImage src={getUserAvatar(comment.userId || '')} />
                      <AvatarFallback className="bg-accentneongreen text-primary">
                        {getUserDisplayName(comment.userId || '').charAt(0)}
                      </AvatarFallback>
                    </Avatar>
                    
                    <div className="flex-1 space-y-3">
                      {/* User Info */}
                      <div className="flex justify-between items-start">
                        <div>
                          <div className="flex items-center space-x-2">
                            <span className="text-sm font-medium text-secondary-foreground">
                              {getUserDisplayName(comment.userId || '')}
                            </span>
                            <div className="flex items-center space-x-1 text-xs text-secondary-foreground/60">
                              <Calendar className="w-3 h-3" />
                              <span>
                                {comment.creationDate 
                                  ? format(new Date(comment.creationDate), 'dd MMM yyyy, HH:mm', { locale: tr })
                                  : 'Tarih bilinmiyor'
                                }
                              </span>
                            </div>
                          </div>
                        </div>
                        {isOwnComment && (
                          <Button
                            size="sm"
                            variant="outline"
                            className="border-red-500/30 text-red-500 hover:bg-red-500 hover:text-white"
                            onClick={() => handleDeleteComment(comment._id, comment.userId || '')}
                          >
                            <Trash2 className="w-3 h-3" />
                          </Button>
                        )}
                      </div>
                      
                      {/* Comment Text */}
                      <p className="text-secondary-foreground leading-relaxed">
                        {comment.commentText}
                      </p>
                      
                      {/* Like/Dislike Buttons */}
                      <div className="flex items-center space-x-4 pt-2">
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => handleLikeToggle(comment._id, 'like')}
                          disabled={likesLoading}
                          className={`text-secondary-foreground/70 hover:text-accentneongreen hover:bg-accentneongreen/10 ${
                            userLike === 'like' ? 'text-accentneongreen bg-accentneongreen/10' : ''
                          }`}
                        >
                          <ThumbsUp className="w-4 h-4 mr-1" />
                          {likeCount > 0 && <span className="text-xs">{likeCount}</span>}
                        </Button>
                        
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => handleLikeToggle(comment._id, 'dislike')}
                          disabled={likesLoading}
                          className={`text-secondary-foreground/70 hover:text-red-400 hover:bg-red-400/10 ${
                            userLike === 'dislike' ? 'text-red-400 bg-red-400/10' : ''
                          }`}
                        >
                          <ThumbsDown className="w-4 h-4 mr-1" />
                          {dislikeCount > 0 && <span className="text-xs">{dislikeCount}</span>}
                        </Button>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            );
          })
        )}
      </div>
    </div>
  );
}