import { useState, useRef, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Play, Pause, Volume2, VolumeX, Maximize, SkipBack, SkipForward, Settings, X, Minimize } from 'lucide-react';
import { VideoStorage } from '@/lib/video-storage';
import { ServerVideoStorage } from '@/lib/server-video-storage';

interface EnhancedVideoPlayerProps {
  src: string;
  poster?: string;
  title?: string;
  onTimeUpdate?: (currentTime: number, duration: number) => void;
  onPlay?: () => void;
  onPause?: () => void;
  className?: string;
}

export function EnhancedVideoPlayer({
  src,
  poster,
  title,
  onTimeUpdate,
  onPlay,
  onPause,
  className = ''
}: EnhancedVideoPlayerProps) {
  const videoRef = useRef<HTMLVideoElement>(null);
  const [isPlaying, setIsPlaying] = useState(false);
  const [currentTime, setCurrentTime] = useState(0);
  const [duration, setDuration] = useState(0);
  const [volume, setVolume] = useState(1);
  const [isMuted, setIsMuted] = useState(false);
  const [isFullscreen, setIsFullscreen] = useState(false);
  const [showControls, setShowControls] = useState(true);
  const [isLoading, setIsLoading] = useState(true);
  const [hasError, setHasError] = useState(false);
  const [playbackRate, setPlaybackRate] = useState(1);
  const [actualVideoSrc, setActualVideoSrc] = useState<string>('');

  const controlsTimeoutRef = useRef<NodeJS.Timeout>();

  // Check if URL is a Mega.nz link
  const isMegaUrl = (url: string) => {
    return url.includes('mega.nz') || url.includes('mega.co.nz');
  };

  // Convert Mega.nz URL to embeddable format
  const convertMegaUrl = (url: string) => {
    try {
      // Extract file ID from various Mega URL formats
      let fileId = '';
      let key = '';
      
      if (url.includes('/file/')) {
        // Format: https://mega.nz/file/FILEID#KEY
        const match = url.match(/\/file\/([^#]+)#(.+)/);
        if (match) {
          fileId = match[1];
          key = match[2];
        }
      } else if (url.includes('#!')) {
        // Format: https://mega.nz/#!FILEID!KEY
        const match = url.match(/#!([^!]+)!(.+)/);
        if (match) {
          fileId = match[1];
          key = match[2];
        }
      }
      
      if (fileId && key) {
        // Try different embed formats
        return {
          embed: `https://mega.nz/embed/${fileId}#${key}`,
          preview: `https://mega.nz/preview/${fileId}#${key}`,
          direct: url
        };
      }
      
      return {
        embed: url,
        preview: url,
        direct: url
      };
    } catch (error) {
      console.error('Mega URL dönüştürme hatası:', error);
      return {
        embed: url,
        preview: url,
        direct: url
      };
    }
  };

  // Load video from various sources
  useEffect(() => {
    const loadVideo = async () => {
      setIsLoading(true);
      setHasError(false);
      
      try {
        if (VideoStorage.isIndexedDBUrl(src)) {
          // Load from IndexedDB
          const videoId = VideoStorage.extractIdFromUrl(src);
          const videoBlob = await VideoStorage.getVideo(videoId);
          if (videoBlob) {
            const blobUrl = URL.createObjectURL(videoBlob);
            setActualVideoSrc(blobUrl);
          } else {
            setHasError(true);
            setIsLoading(false);
          }
        } else if (ServerVideoStorage.isServerVideoUrl(src)) {
          // Load from server storage
          const videoId = ServerVideoStorage.extractVideoIdFromUrl(src);
          const videoUrl = await ServerVideoStorage.getVideoUrl(videoId);
          if (videoUrl) {
            setActualVideoSrc(videoUrl);
          } else {
            setHasError(true);
            setIsLoading(false);
          }
        } else if (isMegaUrl(src)) {
          // Handle Mega.nz URLs - try multiple formats
          const megaUrls = convertMegaUrl(src);
          setActualVideoSrc(megaUrls.embed);
          setIsLoading(false);
        } else {
          // Direct URL
          setActualVideoSrc(src);
        }
      } catch (error) {
        console.error('Video yükleme hatası:', error);
        setHasError(true);
        setIsLoading(false);
      }
    };

    loadVideo();

    // Cleanup blob URL when component unmounts
    return () => {
      if (actualVideoSrc && actualVideoSrc.startsWith('blob:')) {
        URL.revokeObjectURL(actualVideoSrc);
      }
    };
  }, [src]);

  useEffect(() => {
    const video = videoRef.current;
    if (!video) return;

    const handleLoadedMetadata = () => {
      setDuration(video.duration);
      setIsLoading(false);
    };

    const handleTimeUpdate = () => {
      setCurrentTime(video.currentTime);
      onTimeUpdate?.(video.currentTime, video.duration);
    };

    const handlePlay = () => {
      setIsPlaying(true);
      onPlay?.();
    };

    const handlePause = () => {
      setIsPlaying(false);
      onPause?.();
    };

    const handleVolumeChange = () => {
      setVolume(video.volume);
      setIsMuted(video.muted);
    };

    const handleError = () => {
      setHasError(true);
      setIsLoading(false);
    };

    const handleCanPlay = () => {
      setIsLoading(false);
      setHasError(false);
    };

    video.addEventListener('loadedmetadata', handleLoadedMetadata);
    video.addEventListener('timeupdate', handleTimeUpdate);
    video.addEventListener('play', handlePlay);
    video.addEventListener('pause', handlePause);
    video.addEventListener('volumechange', handleVolumeChange);
    video.addEventListener('error', handleError);
    video.addEventListener('canplay', handleCanPlay);

    return () => {
      video.removeEventListener('loadedmetadata', handleLoadedMetadata);
      video.removeEventListener('timeupdate', handleTimeUpdate);
      video.removeEventListener('play', handlePlay);
      video.removeEventListener('pause', handlePause);
      video.removeEventListener('volumechange', handleVolumeChange);
      video.removeEventListener('error', handleError);
      video.removeEventListener('canplay', handleCanPlay);
    };
  }, [onTimeUpdate, onPlay, onPause]);

  const togglePlay = () => {
    const video = videoRef.current;
    if (!video) return;

    if (isPlaying) {
      video.pause();
    } else {
      video.play();
    }
  };

  const handleSeek = (value: number[]) => {
    const video = videoRef.current;
    if (!video) return;

    video.currentTime = value[0];
    setCurrentTime(value[0]);
  };

  const handleVolumeChange = (value: number[]) => {
    const video = videoRef.current;
    if (!video) return;

    const newVolume = value[0];
    video.volume = newVolume;
    setVolume(newVolume);
    setIsMuted(newVolume === 0);
  };

  const toggleMute = () => {
    const video = videoRef.current;
    if (!video) return;

    video.muted = !isMuted;
    setIsMuted(!isMuted);
  };

  const toggleFullscreen = async () => {
    const videoContainer = videoRef.current?.parentElement;
    if (!videoContainer) return;

    try {
      if (!isFullscreen) {
        // Enter fullscreen
        if (videoContainer.requestFullscreen) {
          await videoContainer.requestFullscreen();
        } else if ((videoContainer as any).webkitRequestFullscreen) {
          await (videoContainer as any).webkitRequestFullscreen();
        } else if ((videoContainer as any).mozRequestFullScreen) {
          await (videoContainer as any).mozRequestFullScreen();
        } else if ((videoContainer as any).msRequestFullscreen) {
          await (videoContainer as any).msRequestFullscreen();
        }
      } else {
        // Exit fullscreen
        if (document.exitFullscreen) {
          await document.exitFullscreen();
        } else if ((document as any).webkitExitFullscreen) {
          await (document as any).webkitExitFullscreen();
        } else if ((document as any).mozCancelFullScreen) {
          await (document as any).mozCancelFullScreen();
        } else if ((document as any).msExitFullscreen) {
          await (document as any).msExitFullscreen();
        }
      }
    } catch (error) {
      console.error('Tam ekran modu değiştirilemedi:', error);
    }
  };

  // Listen for fullscreen changes
  useEffect(() => {
    const handleFullscreenChange = () => {
      const isCurrentlyFullscreen = !!(
        document.fullscreenElement ||
        (document as any).webkitFullscreenElement ||
        (document as any).mozFullScreenElement ||
        (document as any).msFullscreenElement
      );
      setIsFullscreen(isCurrentlyFullscreen);
    };

    document.addEventListener('fullscreenchange', handleFullscreenChange);
    document.addEventListener('webkitfullscreenchange', handleFullscreenChange);
    document.addEventListener('mozfullscreenchange', handleFullscreenChange);
    document.addEventListener('MSFullscreenChange', handleFullscreenChange);

    return () => {
      document.removeEventListener('fullscreenchange', handleFullscreenChange);
      document.removeEventListener('webkitfullscreenchange', handleFullscreenChange);
      document.removeEventListener('mozfullscreenchange', handleFullscreenChange);
      document.removeEventListener('MSFullscreenChange', handleFullscreenChange);
    };
  }, []);

  // Handle keyboard shortcuts
  useEffect(() => {
    const handleKeyPress = (e: KeyboardEvent) => {
      if (!videoRef.current) return;

      switch (e.key) {
        case ' ':
          e.preventDefault();
          togglePlay();
          break;
        case 'f':
        case 'F':
          e.preventDefault();
          toggleFullscreen();
          break;
        case 'Escape':
          if (isFullscreen) {
            toggleFullscreen();
          }
          break;
        case 'm':
        case 'M':
          e.preventDefault();
          toggleMute();
          break;
        case 'ArrowLeft':
          e.preventDefault();
          skipTime(-10);
          break;
        case 'ArrowRight':
          e.preventDefault();
          skipTime(10);
          break;
        case 'ArrowUp':
          e.preventDefault();
          const newVolumeUp = Math.min(1, volume + 0.1);
          handleVolumeChange([newVolumeUp]);
          break;
        case 'ArrowDown':
          e.preventDefault();
          const newVolumeDown = Math.max(0, volume - 0.1);
          handleVolumeChange([newVolumeDown]);
          break;
      }
    };

    if (isFullscreen) {
      document.addEventListener('keydown', handleKeyPress);
    }

    return () => {
      document.removeEventListener('keydown', handleKeyPress);
    };
  }, [isFullscreen, isPlaying, volume]);

  const skipTime = (seconds: number) => {
    const video = videoRef.current;
    if (!video) return;

    video.currentTime = Math.max(0, Math.min(video.duration, video.currentTime + seconds));
  };

  const changePlaybackRate = (rate: number) => {
    const video = videoRef.current;
    if (!video) return;

    video.playbackRate = rate;
    setPlaybackRate(rate);
  };

  const formatTime = (seconds: number) => {
    const mins = Math.floor(seconds / 60);
    const secs = Math.floor(seconds % 60);
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  const handleMouseMove = () => {
    setShowControls(true);
    if (controlsTimeoutRef.current) {
      clearTimeout(controlsTimeoutRef.current);
    }
    controlsTimeoutRef.current = setTimeout(() => {
      if (isPlaying) {
        setShowControls(false);
      }
    }, 3000);
  };

  return (
    <div 
      className={`relative bg-black rounded-xl overflow-hidden group ${className} ${
        isFullscreen ? 'fixed inset-0 z-50 rounded-none bg-black' : ''
      }`}
      onMouseMove={handleMouseMove}
      onMouseLeave={() => isPlaying && setShowControls(false)}
    >
      {isMegaUrl(src) ? (
        // Mega.nz embedded video player with fallback
        <div className="relative w-full h-full bg-gradient-to-br from-gray-900 to-black">
          {/* Try iframe first, fallback to custom interface */}
          <iframe
            src={actualVideoSrc}
            className={`w-full h-full border-0 ${isFullscreen ? 'max-h-screen' : ''}`}
            allowFullScreen
            allow="autoplay; encrypted-media; fullscreen"
            title={title || 'Mega.nz Video Player'}
            style={{ minHeight: '300px' }}
            onLoad={() => {
              setIsLoading(false);
              setHasError(false);
            }}
            onError={() => {
              // If iframe fails, show custom interface
              setHasError(false); // Don't show error, show custom interface instead
              setIsLoading(false);
            }}
          />
          
          {/* Custom Mega.nz interface overlay */}
          <div className="absolute inset-0 flex flex-col items-center justify-center text-white p-8 bg-gradient-to-br from-gray-900/95 to-black/95">
            <div className="text-center space-y-6 max-w-md">
              <div className="relative">
                <div className="w-24 h-24 mx-auto bg-accentneongreen/20 rounded-full flex items-center justify-center">
                  <Play className="w-12 h-12 text-accentneongreen" />
                </div>
                <div className="absolute -top-1 -right-1 w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
                  <span className="text-white text-xs font-bold">M</span>
                </div>
              </div>
              
              <div className="space-y-3">
                <h3 className="font-heading text-xl font-bold text-white">
                  Mega.nz Video Oynatıcı
                </h3>
                <p className="font-paragraph text-gray-300 text-sm leading-relaxed">
                  Bu video Mega.nz'de barındırılıyor. Site üzerinde izlemek için aşağıdaki butona tıklayın.
                </p>
              </div>
              
              <div className="space-y-3">
                <Button
                  onClick={() => {
                    // Try to open in a popup window for better integration
                    const popup = window.open(
                      src, 
                      'megavideo', 
                      'width=800,height=600,scrollbars=yes,resizable=yes'
                    );
                    if (!popup) {
                      // Fallback to new tab if popup is blocked
                      window.open(src, '_blank');
                    }
                  }}
                  className="bg-accentneongreen text-black hover:bg-accentneongreen/90 font-medium px-8 py-3 rounded-lg transition-all duration-200 transform hover:scale-105"
                >
                  <Play className="w-5 h-5 mr-2" />
                  Videoyu İzle
                </Button>
                
                <Button
                  onClick={() => window.open(src, '_blank')}
                  variant="outline"
                  className="border-gray-600 text-gray-300 hover:bg-gray-800 hover:border-gray-500 px-6 py-2"
                >
                  Yeni Sekmede Aç
                </Button>
              </div>
              
              <p className="font-paragraph text-gray-500 text-xs">
                Video güvenli bir popup penceresinde açılacak
              </p>
            </div>
          </div>
          
          {/* Fullscreen button */}
          {!isFullscreen && (
            <div className="absolute top-4 right-4 z-10">
              <Button
                onClick={toggleFullscreen}
                size="sm"
                variant="outline"
                className="bg-black/70 border-white/20 text-white hover:bg-black/90 hover:border-accentneongreen"
              >
                <Maximize className="w-4 h-4" />
              </Button>
            </div>
          )}
        </div>
      ) : (
        <video
          ref={videoRef}
          className={`w-full h-full object-contain ${isFullscreen ? 'max-h-screen' : ''}`}
          poster={poster}
          preload="metadata"
          onClick={togglePlay}
          src={actualVideoSrc}
          onLoadStart={() => setIsLoading(true)}
          onCanPlay={() => setIsLoading(false)}
          onError={() => {
            setHasError(true);
            setIsLoading(false);
          }}
        >
          <source src={actualVideoSrc} type="video/mp4" />
          <source src={actualVideoSrc} type="video/webm" />
          <source src={actualVideoSrc} type="video/ogg" />
          <p className="text-white p-4 text-center">
            Tarayıcınız bu video formatını desteklemiyor.
          </p>
        </video>
      )}

      {/* Loading Overlay */}
      {isLoading && (
        <div className="absolute inset-0 bg-black/50 flex items-center justify-center">
          <div className="text-white text-center">
            <div className="w-8 h-8 border-2 border-accentneongreen border-t-transparent rounded-full animate-spin mx-auto mb-2"></div>
            <p className="font-paragraph text-sm">Video yükleniyor...</p>
          </div>
        </div>
      )}

      {/* Error Overlay */}
      {hasError && (
        <div className="absolute inset-0 bg-black/50 flex items-center justify-center">
          <div className="text-white text-center max-w-md p-6">
            <div className="text-red-400 mb-4 text-3xl">⚠️</div>
            <h3 className="font-heading text-lg font-bold mb-2">Video Yüklenemedi</h3>
            {isMegaUrl(src) ? (
              <div className="space-y-3">
                <p className="font-paragraph text-sm text-white/80">
                  Mega.nz videosu yüklenirken bir sorun oluştu. Video doğrudan Mega.nz'de açılabilir.
                </p>
                <Button
                  onClick={() => window.open(src, '_blank')}
                  className="bg-accentneongreen text-black hover:bg-accentneongreen/90 font-medium px-6 py-2"
                >
                  <Play className="w-4 h-4 mr-2" />
                  Mega.nz'de Aç
                </Button>
              </div>
            ) : (
              <div className="space-y-3">
                <p className="font-paragraph text-sm text-white/80">
                  Dosya formatı desteklenmiyor olabilir veya bağlantı geçersiz.
                </p>
                <Button
                  onClick={() => window.open(actualVideoSrc || src, '_blank')}
                  variant="outline"
                  size="sm"
                  className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary"
                >
                  Videoyu İndir
                </Button>
              </div>
            )}
          </div>
        </div>
      )}

      {/* Play Button Overlay - Hide for Mega iframe */}
      {!isPlaying && !isLoading && !hasError && !isMegaUrl(src) && (
        <div className="absolute inset-0 flex items-center justify-center">
          <Button
            onClick={togglePlay}
            size="lg"
            className="bg-accentneongreen/90 hover:bg-accentneongreen text-primary rounded-full w-16 h-16 p-0"
          >
            <Play className="w-8 h-8 ml-1" />
          </Button>
        </div>
      )}

      {/* Controls - Hide for Mega iframe */}
      {showControls && !isLoading && !hasError && !isMegaUrl(src) && (
        <div className="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/80 to-transparent p-4">
          {/* Progress Bar */}
          <div className="mb-4">
            <div className="relative w-full h-1 bg-white/20 rounded-full cursor-pointer"
                 onClick={(e) => {
                   const rect = e.currentTarget.getBoundingClientRect();
                   const percent = (e.clientX - rect.left) / rect.width;
                   const newTime = percent * duration;
                   handleSeek([newTime]);
                 }}>
              <div 
                className="absolute h-full bg-accentneongreen rounded-full"
                style={{ width: `${(currentTime / duration) * 100}%` }}
              />
              <div 
                className="absolute w-3 h-3 bg-accentneongreen rounded-full -mt-1 cursor-pointer"
                style={{ left: `${(currentTime / duration) * 100}%` }}
              />
            </div>
            <div className="flex justify-between text-white text-xs mt-1">
              <span>{formatTime(currentTime)}</span>
              <span>{formatTime(duration)}</span>
            </div>
          </div>

          {/* Control Buttons */}
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-2">
              <Button
                onClick={() => skipTime(-10)}
                variant="ghost"
                size="sm"
                className="text-white hover:text-accentneongreen hover:bg-white/10"
              >
                <SkipBack className="w-4 h-4" />
              </Button>
              
              <Button
                onClick={togglePlay}
                variant="ghost"
                size="sm"
                className="text-white hover:text-accentneongreen hover:bg-white/10"
              >
                {isPlaying ? <Pause className="w-5 h-5" /> : <Play className="w-5 h-5" />}
              </Button>
              
              <Button
                onClick={() => skipTime(10)}
                variant="ghost"
                size="sm"
                className="text-white hover:text-accentneongreen hover:bg-white/10"
              >
                <SkipForward className="w-4 h-4" />
              </Button>

              <div className="flex items-center space-x-2 ml-4">
                <Button
                  onClick={toggleMute}
                  variant="ghost"
                  size="sm"
                  className="text-white hover:text-accentneongreen hover:bg-white/10"
                >
                  {isMuted ? <VolumeX className="w-4 h-4" /> : <Volume2 className="w-4 h-4" />}
                </Button>
                
                <div className="w-20">
                  <div className="relative w-full h-1 bg-white/20 rounded-full cursor-pointer"
                       onClick={(e) => {
                         const rect = e.currentTarget.getBoundingClientRect();
                         const percent = (e.clientX - rect.left) / rect.width;
                         handleVolumeChange([percent]);
                       }}>
                    <div 
                      className="absolute h-full bg-accentneongreen rounded-full"
                      style={{ width: `${(isMuted ? 0 : volume) * 100}%` }}
                    />
                    <div 
                      className="absolute w-2 h-2 bg-accentneongreen rounded-full -mt-0.5 cursor-pointer"
                      style={{ left: `${(isMuted ? 0 : volume) * 100}%` }}
                    />
                  </div>
                </div>
              </div>
            </div>

            <div className="flex items-center space-x-2">
              <select
                value={playbackRate}
                onChange={(e) => changePlaybackRate(Number(e.target.value))}
                className="bg-black/50 text-white text-xs rounded px-2 py-1 border border-white/20 focus:outline-none focus:border-accentneongreen"
              >
                <option value={0.5}>0.5x</option>
                <option value={0.75}>0.75x</option>
                <option value={1}>Normal</option>
                <option value={1.25}>1.25x</option>
                <option value={1.5}>1.5x</option>
                <option value={2}>2x</option>
              </select>

              <Button
                onClick={toggleFullscreen}
                variant="ghost"
                size="sm"
                className="text-white hover:text-accentneongreen hover:bg-white/10"
                title={isFullscreen ? "Tam Ekrandan Çık (ESC)" : "Tam Ekran (F)"}
              >
                {isFullscreen ? <Minimize className="w-4 h-4" /> : <Maximize className="w-4 h-4" />}
              </Button>
            </div>
          </div>
        </div>
      )}

      {/* Mega.nz Info Overlay */}
      {isMegaUrl(src) && !isLoading && (
        <div className="absolute bottom-4 left-4 right-4">
          <div className="bg-black/70 backdrop-blur-sm rounded-lg px-4 py-3 border border-accentneongreen/30">
            <p className="text-white text-sm font-paragraph">
              <span className="text-accentneongreen font-medium">Mega.nz Video:</span> Bu video Mega.nz üzerinden gömülü olarak oynatılıyor.
            </p>
          </div>
        </div>
      )}

      {/* Title Overlay */}
      {title && showControls && !isLoading && !isFullscreen && !isMegaUrl(src) && (
        <div className="absolute top-4 left-4 right-4">
          <h3 className="text-white font-heading text-lg font-bold bg-black/50 backdrop-blur-sm rounded-lg px-3 py-2">
            {title}
          </h3>
        </div>
      )}

      {/* Fullscreen Title */}
      {title && isFullscreen && showControls && !isLoading && !isMegaUrl(src) && (
        <div className="absolute bottom-20 left-4 right-4 z-10">
          <h3 className="text-white font-heading text-2xl font-bold bg-black/70 backdrop-blur-sm rounded-lg px-4 py-3 border border-white/20">
            {title}
          </h3>
        </div>
      )}

      {/* Fullscreen Exit Button - Top Right */}
      {isFullscreen && (
        <div className="absolute top-4 right-4 z-20">
          <Button
            onClick={toggleFullscreen}
            variant="ghost"
            size="sm"
            className="text-white hover:text-accentneongreen hover:bg-black/70 bg-black/50 backdrop-blur-sm border border-white/20"
            title="Tam Ekrandan Çık (ESC)"
          >
            <X className="w-5 h-5" />
          </Button>
        </div>
      )}

      {/* Fullscreen Instructions */}
      {isFullscreen && showControls && !isMegaUrl(src) && (
        <div className="absolute top-4 left-4 z-10">
          <div className="bg-black/70 backdrop-blur-sm rounded-lg px-3 py-2 border border-white/20">
            <p className="text-white text-sm font-paragraph">
              <span className="text-accentneongreen font-medium">Klavye Kısayolları:</span> Boşluk (Oynat/Duraklat) • F (Tam Ekran) • ESC (Çık) • M (Ses) • ←/→ (İleri/Geri)
            </p>
          </div>
        </div>
      )}
    </div>
  );
}