import { useState, useRef } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Progress } from '@/components/ui/progress';
import { Switch } from '@/components/ui/switch';
import { Upload, X, CheckCircle, AlertCircle, FileText, BookOpen } from 'lucide-react';
import { BaseCrudService } from '@/integrations';
import { MangaChapters } from '@/entities';

interface MangaChapterUploadDialogProps {
  isOpen: boolean;
  onClose: () => void;
  mangaId: string;
  mangaTitle?: string;
  onChapterAdded?: () => void;
}

export function MangaChapterUploadDialog({ 
  isOpen, 
  onClose, 
  mangaId, 
  mangaTitle,
  onChapterAdded 
}: MangaChapterUploadDialogProps) {
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [chapterNumber, setChapterNumber] = useState('');
  const [chapterTitle, setChapterTitle] = useState('');
  const [releaseDate, setReleaseDate] = useState('');
  const [isPublished, setIsPublished] = useState(true);
  const [isUploading, setIsUploading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [uploadStatus, setUploadStatus] = useState<'idle' | 'uploading' | 'success' | 'error'>('idle');
  const [errorMessage, setErrorMessage] = useState('');
  const fileInputRef = useRef<HTMLInputElement>(null);

  const handleFileSelect = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      // Check if file is a PDF
      if (file.type !== 'application/pdf') {
        setErrorMessage('Lütfen geçerli bir PDF dosyası seçin.');
        return;
      }
      
      // Check file size (max 100MB for chapters)
      const maxSize = 100 * 1024 * 1024; // 100MB
      if (file.size > maxSize) {
        setErrorMessage('PDF dosyası 100MB\'dan küçük olmalıdır.');
        return;
      }

      setSelectedFile(file);
      setErrorMessage('');
    }
  };

  const simulateUpload = async () => {
    setIsUploading(true);
    setUploadStatus('uploading');
    setUploadProgress(0);

    // Simulate upload progress
    for (let i = 0; i <= 100; i += 4) {
      await new Promise(resolve => setTimeout(resolve, 60));
      setUploadProgress(i);
    }

    try {
      // Create a blob URL for the PDF file that can be accessed
      const pdfBlob = new Blob([selectedFile!], { type: selectedFile!.type });
      const pdfUrl = URL.createObjectURL(pdfBlob);
      
      // For demo purposes, we'll use a data URL that can actually be accessed
      const reader = new FileReader();
      reader.onload = async (e) => {
        const dataUrl = e.target?.result as string;
        
        // Create chapter entry in database with accessible URL
        const newChapter = {
          _id: crypto.randomUUID(),
          mangaReference: mangaId,
          chapterNumber: chapterNumber ? parseInt(chapterNumber) : undefined,
          chapterTitle: chapterTitle,
          pdfUrl: dataUrl, // Use data URL for immediate access
          pdfFile: dataUrl, // Use data URL for immediate access
          chapterContentUrl: dataUrl, // Use data URL for immediate access
          releaseDate: releaseDate ? new Date(releaseDate).toISOString() : new Date().toISOString(),
          isPublished: isPublished
        };

        await BaseCrudService.create('mangachapters', newChapter as any);
        
        setUploadStatus('success');
        
        // Call callback to refresh parent component
        if (onChapterAdded) {
          onChapterAdded();
        }
        
        // Reset form after successful upload
        setTimeout(() => {
          resetForm();
          onClose();
        }, 2000);
      };
      
      reader.readAsDataURL(selectedFile!);
      
    } catch (error) {
      console.error('Upload error:', error);
      setUploadStatus('error');
      setErrorMessage('Bölüm yüklenirken bir hata oluştu. Lütfen tekrar deneyin.');
      setIsUploading(false);
    }
  };

  const resetForm = () => {
    setSelectedFile(null);
    setChapterNumber('');
    setChapterTitle('');
    setReleaseDate('');
    setIsPublished(true);
    setUploadProgress(0);
    setUploadStatus('idle');
    setErrorMessage('');
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const handleClose = () => {
    if (!isUploading) {
      resetForm();
      onClose();
    }
  };

  const canSubmit = selectedFile && chapterNumber.trim() && chapterTitle.trim() && !isUploading;

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-[700px] bg-secondary border-0 max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="font-heading text-2xl font-bold text-secondary-foreground flex items-center gap-2">
            <BookOpen className="w-6 h-6 text-accentneongreen" />
            Manga Bölümü Yükle
          </DialogTitle>
          {mangaTitle && (
            <p className="font-paragraph text-secondary-foreground/70 text-sm mt-1">
              {mangaTitle} için yeni bölüm ekleniyor
            </p>
          )}
        </DialogHeader>

        <div className="space-y-6 py-4">
          {/* File Upload Area */}
          <div className="space-y-2">
            <Label className="font-paragraph text-secondary-foreground font-medium">PDF Dosyası</Label>
            <div 
              className={`border-2 border-dashed rounded-lg p-6 text-center transition-colors ${
                selectedFile 
                  ? 'border-accentneongreen bg-accentneongreen/5' 
                  : 'border-secondary-foreground/30 hover:border-accentneongreen/50'
              }`}
            >
              {selectedFile ? (
                <div className="space-y-2">
                  <FileText className="w-10 h-10 text-accentneongreen mx-auto" />
                  <p className="font-paragraph text-secondary-foreground font-medium">
                    {selectedFile.name}
                  </p>
                  <p className="font-paragraph text-secondary-foreground/60 text-sm">
                    {(selectedFile.size / (1024 * 1024)).toFixed(2)} MB
                  </p>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => {
                      setSelectedFile(null);
                      if (fileInputRef.current) fileInputRef.current.value = '';
                    }}
                    className="border-secondary-foreground/30 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
                  >
                    <X className="w-4 h-4 mr-2" />
                    Kaldır
                  </Button>
                </div>
              ) : (
                <div className="space-y-2">
                  <Upload className="w-10 h-10 text-secondary-foreground/40 mx-auto" />
                  <p className="font-paragraph text-secondary-foreground">
                    Bölüm PDF'ini buraya sürükleyin veya seçin
                  </p>
                  <p className="font-paragraph text-secondary-foreground/60 text-sm">
                    Sadece PDF formatı desteklenir (Max: 100MB)
                  </p>
                  <Button
                    variant="outline"
                    onClick={() => fileInputRef.current?.click()}
                    className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary"
                  >
                    Dosya Seç
                  </Button>
                </div>
              )}
              <input
                ref={fileInputRef}
                type="file"
                accept=".pdf,application/pdf"
                onChange={handleFileSelect}
                className="hidden"
              />
            </div>
          </div>

          {/* Form Fields */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="chapterNumber" className="font-paragraph text-secondary-foreground font-medium">
                Bölüm Numarası *
              </Label>
              <Input
                id="chapterNumber"
                type="number"
                value={chapterNumber}
                onChange={(e) => setChapterNumber(e.target.value)}
                placeholder="1"
                min="1"
                className="bg-primary border-secondary-foreground/30 text-primary-foreground"
                disabled={isUploading}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="releaseDate" className="font-paragraph text-secondary-foreground font-medium">
                Yayın Tarihi
              </Label>
              <Input
                id="releaseDate"
                type="date"
                value={releaseDate}
                onChange={(e) => setReleaseDate(e.target.value)}
                className="bg-primary border-secondary-foreground/30 text-primary-foreground"
                disabled={isUploading}
              />
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="chapterTitle" className="font-paragraph text-secondary-foreground font-medium">
              Bölüm Başlığı *
            </Label>
            <Input
              id="chapterTitle"
              value={chapterTitle}
              onChange={(e) => setChapterTitle(e.target.value)}
              placeholder="Bölüm başlığını girin"
              className="bg-primary border-secondary-foreground/30 text-primary-foreground"
              disabled={isUploading}
            />
          </div>

          {/* Publish Status */}
          <div className="flex items-center justify-between p-4 bg-primary/50 rounded-lg">
            <div className="space-y-1">
              <Label htmlFor="isPublished" className="font-paragraph text-secondary-foreground font-medium">
                Hemen Yayınla
              </Label>
              <p className="font-paragraph text-secondary-foreground/60 text-sm">
                Bölüm yüklendikten sonra okuyucular tarafından görülebilir olsun
              </p>
            </div>
            <Switch
              id="isPublished"
              checked={isPublished}
              onCheckedChange={setIsPublished}
              disabled={isUploading}
            />
          </div>

          {/* Upload Progress */}
          {uploadStatus === 'uploading' && (
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <Label className="font-paragraph text-secondary-foreground font-medium">
                  Yükleme İlerlemesi
                </Label>
                <span className="font-paragraph text-secondary-foreground/60 text-sm">
                  %{uploadProgress}
                </span>
              </div>
              <Progress value={uploadProgress} className="h-2" />
              <p className="font-paragraph text-secondary-foreground/60 text-sm">
                Bölüm yükleniyor, lütfen bekleyin...
              </p>
            </div>
          )}

          {/* Success Message */}
          {uploadStatus === 'success' && (
            <div className="flex items-center gap-2 p-4 bg-green-500/10 border border-green-500/30 rounded-lg">
              <CheckCircle className="w-5 h-5 text-green-400" />
              <p className="font-paragraph text-green-400">
                Bölüm başarıyla yüklendi! Manga koleksiyonuna eklendi.
              </p>
            </div>
          )}

          {/* Error Message */}
          {(uploadStatus === 'error' || errorMessage) && (
            <div className="flex items-center gap-2 p-4 bg-red-500/10 border border-red-500/30 rounded-lg">
              <AlertCircle className="w-5 h-5 text-red-400" />
              <p className="font-paragraph text-red-400">
                {errorMessage || 'Bir hata oluştu. Lütfen tekrar deneyin.'}
              </p>
            </div>
          )}

          {/* Action Buttons */}
          <div className="flex justify-end gap-3 pt-4">
            <Button
              variant="outline"
              onClick={handleClose}
              disabled={isUploading}
              className="border-secondary-foreground/30 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
            >
              İptal
            </Button>
            <Button
              onClick={simulateUpload}
              disabled={!canSubmit}
              className="bg-accentneongreen text-primary hover:bg-accentneongreen/90 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isUploading ? (
                <>
                  <div className="w-4 h-4 border-2 border-primary border-t-transparent rounded-full animate-spin mr-2" />
                  Yükleniyor...
                </>
              ) : (
                <>
                  <Upload className="w-4 h-4 mr-2" />
                  Bölüm Yükle
                </>
              )}
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}