import React, { useState } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Maximize2, Minimize2, ExternalLink, Calendar, Book, User, X } from 'lucide-react';
import { MangaTitles } from '@/entities';
import { format } from 'date-fns';
import { tr } from 'date-fns/locale';

interface MangaPreviewProps {
  manga: MangaTitles;
  isOpen: boolean;
  onClose: () => void;
}

export function MangaPreview({ manga, isOpen, onClose }: MangaPreviewProps) {
  const [isFullscreen, setIsFullscreen] = useState(false);

  const toggleFullscreen = () => {
    setIsFullscreen(!isFullscreen);
  };

  const openExternalLink = () => {
    if (manga.pdfUrl) {
      window.open(manga.pdfUrl, '_blank', 'noopener,noreferrer');
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent 
        className={`bg-gray-900 border-gray-700 text-white ${
          isFullscreen 
            ? 'max-w-[100vw] max-h-[100vh] w-[100vw] h-[100vh] rounded-none' 
            : 'max-w-4xl max-h-[90vh]'
        } overflow-hidden`}
      >
        {/* Header */}
        <DialogHeader className="flex flex-row items-center justify-between space-y-0 pb-4 border-b border-gray-700">
          <DialogTitle className="text-xl font-bold text-white truncate pr-4">
            {manga.title || 'Manga Önizleme'}
          </DialogTitle>
          <div className="flex items-center space-x-2 flex-shrink-0">
            <Button
              variant="ghost"
              size="sm"
              onClick={toggleFullscreen}
              className="text-gray-400 hover:text-white hover:bg-gray-800"
            >
              {isFullscreen ? (
                <Minimize2 className="w-4 h-4" />
              ) : (
                <Maximize2 className="w-4 h-4" />
              )}
            </Button>
            <Button
              variant="ghost"
              size="sm"
              onClick={onClose}
              className="text-gray-400 hover:text-white hover:bg-gray-800"
            >
              <X className="w-4 h-4" />
            </Button>
          </div>
        </DialogHeader>

        {/* Content */}
        <div className={`${isFullscreen ? 'h-[calc(100vh-80px)]' : 'max-h-[70vh]'} overflow-y-auto`}>
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 p-6">
            {/* Cover Image */}
            <div className="lg:col-span-1">
              <div className="relative aspect-[3/4] rounded-lg overflow-hidden bg-gray-800">
                {manga.coverImage ? (
                  <img
                    src={manga.coverImage}
                    alt={manga.title || 'Manga kapağı'}
                    className="w-full h-full object-cover"
                  />
                ) : (
                  <div className="w-full h-full flex items-center justify-center">
                    <Book className="w-16 h-16 text-gray-600" />
                  </div>
                )}
                
                {/* Featured Badge */}
                {manga.isFeatured && (
                  <Badge className="absolute top-3 left-3 bg-accentneongreen text-black">
                    ⭐ Öne Çıkan
                  </Badge>
                )}
                
                {/* Genre Badge */}
                {manga.genre && (
                  <Badge className="absolute top-3 right-3 bg-black/80 text-white">
                    {manga.genre}
                  </Badge>
                )}
              </div>

              {/* Action Buttons */}
              <div className="mt-4 space-y-2">
                {manga.pdfUrl ? (
                  <Button
                    onClick={openExternalLink}
                    className="w-full bg-accentneongreen text-black hover:bg-accentneongreen/80"
                  >
                    <ExternalLink className="w-4 h-4 mr-2" />
                    Harici Linkten Oku
                  </Button>
                ) : (
                  <Button
                    disabled
                    className="w-full bg-gray-700 text-gray-400 cursor-not-allowed"
                  >
                    <Book className="w-4 h-4 mr-2" />
                    Link Mevcut Değil
                  </Button>
                )}
              </div>
            </div>

            {/* Details */}
            <div className="lg:col-span-2 space-y-6">
              {/* Title and Basic Info */}
              <div>
                <h1 className="text-3xl font-bold text-white mb-2">
                  {manga.title || 'Başlık Yok'}
                </h1>
                
                {/* Meta Information */}
                <div className="flex flex-wrap gap-4 text-sm text-gray-400">
                  {manga.author && (
                    <div className="flex items-center space-x-1">
                      <User className="w-4 h-4" />
                      <span>{manga.author}</span>
                    </div>
                  )}
                  
                  {manga.publicationDate && (
                    <div className="flex items-center space-x-1">
                      <Calendar className="w-4 h-4" />
                      <span>
                        {format(new Date(manga.publicationDate), 'yyyy', { locale: tr })}
                      </span>
                    </div>
                  )}
                  
                  {manga.numberOfChapters && (
                    <div className="flex items-center space-x-1">
                      <Book className="w-4 h-4" />
                      <span>{manga.numberOfChapters} Bölüm</span>
                    </div>
                  )}
                </div>
              </div>

              {/* Synopsis */}
              <div>
                <h3 className="text-lg font-semibold text-white mb-3">Özet</h3>
                <p className="text-gray-300 leading-relaxed">
                  {manga.synopsis || 'Bu manga için henüz bir özet eklenmemiş.'}
                </p>
              </div>

              {/* Additional Info */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="bg-gray-800 rounded-lg p-4">
                  <h4 className="font-semibold text-white mb-2">Tür</h4>
                  <p className="text-gray-300">{manga.genre || 'Belirtilmemiş'}</p>
                </div>
                
                <div className="bg-gray-800 rounded-lg p-4">
                  <h4 className="font-semibold text-white mb-2">Yayın Tarihi</h4>
                  <p className="text-gray-300">
                    {manga.publicationDate 
                      ? format(new Date(manga.publicationDate), 'dd MMMM yyyy', { locale: tr })
                      : 'Belirtilmemiş'
                    }
                  </p>
                </div>
                
                <div className="bg-gray-800 rounded-lg p-4">
                  <h4 className="font-semibold text-white mb-2">Bölüm Sayısı</h4>
                  <p className="text-gray-300">{manga.numberOfChapters || 'Bilinmiyor'}</p>
                </div>
                
                <div className="bg-gray-800 rounded-lg p-4">
                  <h4 className="font-semibold text-white mb-2">Durum</h4>
                  <p className="text-gray-300">
                    {manga.isFeatured ? 'Öne Çıkan' : 'Normal'}
                  </p>
                </div>
              </div>

              {/* Reading Options */}
              <div className="bg-gray-800 rounded-lg p-4">
                <h4 className="font-semibold text-white mb-3">Okuma Seçenekleri</h4>
                <div className="space-y-2">
                  {manga.pdfUrl ? (
                    <div className="flex items-center justify-between p-3 bg-gray-700 rounded">
                      <div>
                        <p className="text-white font-medium">Harici Link</p>
                        <p className="text-sm text-gray-400">PDF formatında okuyun</p>
                      </div>
                      <Button
                        onClick={openExternalLink}
                        size="sm"
                        className="bg-accentneongreen text-black hover:bg-accentneongreen/80"
                      >
                        <ExternalLink className="w-4 h-4 mr-2" />
                        Aç
                      </Button>
                    </div>
                  ) : (
                    <div className="flex items-center justify-between p-3 bg-gray-700 rounded opacity-50">
                      <div>
                        <p className="text-gray-400 font-medium">Okuma Linki</p>
                        <p className="text-sm text-gray-500">Henüz mevcut değil</p>
                      </div>
                      <Button
                        disabled
                        size="sm"
                        className="bg-gray-600 text-gray-400 cursor-not-allowed"
                      >
                        Mevcut Değil
                      </Button>
                    </div>
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}