import { useState, useRef } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Progress } from '@/components/ui/progress';
import { Switch } from '@/components/ui/switch';
import { Upload, X, CheckCircle, AlertCircle, FileText, BookOpen } from 'lucide-react';
import { BaseCrudService } from '@/integrations';
import { MangaTitles } from '@/entities';

interface MangaTitleUploadDialogProps {
  isOpen: boolean;
  onClose: () => void;
  onMangaAdded?: () => void;
}

export function MangaTitleUploadDialog({ 
  isOpen, 
  onClose, 
  onMangaAdded 
}: MangaTitleUploadDialogProps) {
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [title, setTitle] = useState('');
  const [synopsis, setSynopsis] = useState('');
  const [author, setAuthor] = useState('');
  const [genre, setGenre] = useState('');
  const [publicationDate, setPublicationDate] = useState('');
  const [numberOfChapters, setNumberOfChapters] = useState('');
  const [isFeatured, setIsFeatured] = useState(false);
  const [isUploading, setIsUploading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [uploadStatus, setUploadStatus] = useState<'idle' | 'uploading' | 'success' | 'error'>('idle');
  const [errorMessage, setErrorMessage] = useState('');
  const fileInputRef = useRef<HTMLInputElement>(null);

  const handleFileSelect = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      // Check if file is an image
      if (!file.type.startsWith('image/')) {
        setErrorMessage('Lütfen geçerli bir görsel dosyası seçin.');
        return;
      }
      
      // Check file size (max 10MB for cover images)
      const maxSize = 10 * 1024 * 1024; // 10MB
      if (file.size > maxSize) {
        setErrorMessage('Görsel dosyası 10MB\'dan küçük olmalıdır.');
        return;
      }

      setSelectedFile(file);
      setErrorMessage('');
    }
  };

  const simulateUpload = async () => {
    setIsUploading(true);
    setUploadStatus('uploading');
    setUploadProgress(0);

    // Simulate upload progress
    for (let i = 0; i <= 100; i += 5) {
      await new Promise(resolve => setTimeout(resolve, 100));
      setUploadProgress(i);
    }

    try {
      // Create manga entry in database
      const newManga = {
        _id: crypto.randomUUID(),
        title: title,
        synopsis: synopsis,
        author: author,
        genre: genre,
        publicationDate: publicationDate ? new Date(publicationDate).toISOString() : undefined,
        numberOfChapters: numberOfChapters ? parseInt(numberOfChapters) : undefined,
        coverImage: selectedFile ? 'https://static.wixstatic.com/media/a92b31_8ddc2dcfa22048faacb33b8ac532b69a~mv2.png?originWidth=320&originHeight=320' : undefined,
        coverImageUrl: selectedFile ? 'https://static.wixstatic.com/media/a92b31_3f9ce960a016482ebc01c078593b21c9~mv2.png?originWidth=320&originHeight=320' : undefined,
        isFeatured: isFeatured
      };

      await BaseCrudService.create('mangatitles', newManga as any);
      
      setUploadStatus('success');
      
      // Call callback to refresh parent component
      if (onMangaAdded) {
        onMangaAdded();
      }
      
      // Reset form after successful upload
      setTimeout(() => {
        resetForm();
        onClose();
      }, 2000);
      
    } catch (error) {
      console.error('Upload error:', error);
      setUploadStatus('error');
      setErrorMessage('Manga yüklenirken bir hata oluştu. Lütfen tekrar deneyin.');
    } finally {
      setIsUploading(false);
    }
  };

  const resetForm = () => {
    setSelectedFile(null);
    setTitle('');
    setSynopsis('');
    setAuthor('');
    setGenre('');
    setPublicationDate('');
    setNumberOfChapters('');
    setIsFeatured(false);
    setUploadProgress(0);
    setUploadStatus('idle');
    setErrorMessage('');
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const handleClose = () => {
    if (!isUploading) {
      resetForm();
      onClose();
    }
  };

  const canSubmit = title.trim() && synopsis.trim() && !isUploading;

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-[700px] bg-secondary border-0 max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="font-heading text-2xl font-bold text-secondary-foreground flex items-center gap-2">
            <BookOpen className="w-6 h-6 text-accentneongreen" />
            Yeni Manga Ekle
          </DialogTitle>
        </DialogHeader>

        <div className="space-y-6 py-4">
          {/* Cover Image Upload Area */}
          <div className="space-y-2">
            <Label className="font-paragraph text-secondary-foreground font-medium">Kapak Görseli</Label>
            <div 
              className={`border-2 border-dashed rounded-lg p-6 text-center transition-colors ${
                selectedFile 
                  ? 'border-accentneongreen bg-accentneongreen/5' 
                  : 'border-secondary-foreground/30 hover:border-accentneongreen/50'
              }`}
            >
              {selectedFile ? (
                <div className="space-y-2">
                  <div className="w-20 h-20 mx-auto rounded-lg overflow-hidden">
                    <img 
                      src={URL.createObjectURL(selectedFile)} 
                      alt="Kapak önizleme"
                      className="w-full h-full object-cover"
                    />
                  </div>
                  <p className="font-paragraph text-secondary-foreground font-medium">
                    {selectedFile.name}
                  </p>
                  <p className="font-paragraph text-secondary-foreground/60 text-sm">
                    {(selectedFile.size / (1024 * 1024)).toFixed(2)} MB
                  </p>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => {
                      setSelectedFile(null);
                      if (fileInputRef.current) fileInputRef.current.value = '';
                    }}
                    className="border-secondary-foreground/30 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
                  >
                    <X className="w-4 h-4 mr-2" />
                    Kaldır
                  </Button>
                </div>
              ) : (
                <div className="space-y-2">
                  <Upload className="w-10 h-10 text-secondary-foreground/40 mx-auto" />
                  <p className="font-paragraph text-secondary-foreground">
                    Kapak görselini buraya sürükleyin veya seçin
                  </p>
                  <p className="font-paragraph text-secondary-foreground/60 text-sm">
                    JPG, PNG formatları desteklenir (Max: 10MB)
                  </p>
                  <Button
                    variant="outline"
                    onClick={() => fileInputRef.current?.click()}
                    className="border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary"
                  >
                    Görsel Seç
                  </Button>
                </div>
              )}
              <input
                ref={fileInputRef}
                type="file"
                accept="image/*"
                onChange={handleFileSelect}
                className="hidden"
              />
            </div>
          </div>

          {/* Form Fields */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="title" className="font-paragraph text-secondary-foreground font-medium">
                Manga Başlığı *
              </Label>
              <Input
                id="title"
                value={title}
                onChange={(e) => setTitle(e.target.value)}
                placeholder="Manga başlığını girin"
                className="bg-primary border-secondary-foreground/30 text-primary-foreground"
                disabled={isUploading}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="author" className="font-paragraph text-secondary-foreground font-medium">
                Yazar
              </Label>
              <Input
                id="author"
                value={author}
                onChange={(e) => setAuthor(e.target.value)}
                placeholder="Yazar adını girin"
                className="bg-primary border-secondary-foreground/30 text-primary-foreground"
                disabled={isUploading}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="genre" className="font-paragraph text-secondary-foreground font-medium">
                Tür
              </Label>
              <Input
                id="genre"
                value={genre}
                onChange={(e) => setGenre(e.target.value)}
                placeholder="Manga türünü girin"
                className="bg-primary border-secondary-foreground/30 text-primary-foreground"
                disabled={isUploading}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="publicationDate" className="font-paragraph text-secondary-foreground font-medium">
                Yayın Tarihi
              </Label>
              <Input
                id="publicationDate"
                type="date"
                value={publicationDate}
                onChange={(e) => setPublicationDate(e.target.value)}
                className="bg-primary border-secondary-foreground/30 text-primary-foreground"
                disabled={isUploading}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="numberOfChapters" className="font-paragraph text-secondary-foreground font-medium">
                Bölüm Sayısı
              </Label>
              <Input
                id="numberOfChapters"
                type="number"
                value={numberOfChapters}
                onChange={(e) => setNumberOfChapters(e.target.value)}
                placeholder="12"
                min="1"
                className="bg-primary border-secondary-foreground/30 text-primary-foreground"
                disabled={isUploading}
              />
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="synopsis" className="font-paragraph text-secondary-foreground font-medium">
              Özet *
            </Label>
            <Textarea
              id="synopsis"
              value={synopsis}
              onChange={(e) => setSynopsis(e.target.value)}
              placeholder="Manga hakkında kısa bir açıklama yazın..."
              rows={4}
              className="bg-primary border-secondary-foreground/30 text-primary-foreground resize-none"
              disabled={isUploading}
            />
          </div>

          {/* Featured Status */}
          <div className="flex items-center justify-between p-4 bg-primary/50 rounded-lg">
            <div className="space-y-1">
              <Label htmlFor="isFeatured" className="font-paragraph text-secondary-foreground font-medium">
                Öne Çıkan Manga
              </Label>
              <p className="font-paragraph text-secondary-foreground/60 text-sm">
                Bu manga ana sayfada öne çıkan bölümde görüntülensin
              </p>
            </div>
            <Switch
              id="isFeatured"
              checked={isFeatured}
              onCheckedChange={setIsFeatured}
              disabled={isUploading}
            />
          </div>

          {/* Upload Progress */}
          {uploadStatus === 'uploading' && (
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <Label className="font-paragraph text-secondary-foreground font-medium">
                  Yükleme İlerlemesi
                </Label>
                <span className="font-paragraph text-secondary-foreground/60 text-sm">
                  %{uploadProgress}
                </span>
              </div>
              <Progress value={uploadProgress} className="h-2" />
              <p className="font-paragraph text-secondary-foreground/60 text-sm">
                Manga yükleniyor, lütfen bekleyin...
              </p>
            </div>
          )}

          {/* Success Message */}
          {uploadStatus === 'success' && (
            <div className="flex items-center gap-2 p-4 bg-green-500/10 border border-green-500/30 rounded-lg">
              <CheckCircle className="w-5 h-5 text-green-400" />
              <p className="font-paragraph text-green-400">
                Manga başarıyla eklendi! Koleksiyona eklendi.
              </p>
            </div>
          )}

          {/* Error Message */}
          {(uploadStatus === 'error' || errorMessage) && (
            <div className="flex items-center gap-2 p-4 bg-red-500/10 border border-red-500/30 rounded-lg">
              <AlertCircle className="w-5 h-5 text-red-400" />
              <p className="font-paragraph text-red-400">
                {errorMessage || 'Bir hata oluştu. Lütfen tekrar deneyin.'}
              </p>
            </div>
          )}

          {/* Action Buttons */}
          <div className="flex justify-end gap-3 pt-4">
            <Button
              variant="outline"
              onClick={handleClose}
              disabled={isUploading}
              className="border-secondary-foreground/30 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
            >
              İptal
            </Button>
            <Button
              onClick={simulateUpload}
              disabled={!canSubmit}
              className="bg-accentneongreen text-primary hover:bg-accentneongreen/90 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isUploading ? (
                <>
                  <div className="w-4 h-4 border-2 border-primary border-t-transparent rounded-full animate-spin mr-2" />
                  Yükleniyor...
                </>
              ) : (
                <>
                  <Upload className="w-4 h-4 mr-2" />
                  Manga Ekle
                </>
              )}
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}