import React, { useState, useEffect } from 'react';
import { X, ExternalLink } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { BaseCrudService } from '@/integrations';
import { Notifications } from '@/entities';

interface NotificationBannerProps {
  className?: string;
}

export function NotificationBanner({ className = '' }: NotificationBannerProps) {
  const [notifications, setNotifications] = useState<Notifications[]>([]);
  const [dismissedNotifications, setDismissedNotifications] = useState<string[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchNotifications();
    // Load dismissed notifications from localStorage
    const dismissed = localStorage.getItem('dismissedNotifications');
    if (dismissed) {
      setDismissedNotifications(JSON.parse(dismissed));
    }
  }, []);

  const fetchNotifications = async () => {
    try {
      const { items } = await BaseCrudService.getAll<Notifications>('notifications');
      // Filter active notifications and sort by display order
      const activeNotifications = items
        .filter(notification => notification.isActive)
        .sort((a, b) => (a.displayOrder || 0) - (b.displayOrder || 0));
      setNotifications(activeNotifications);
    } catch (error) {
      console.error('Bildirimler yüklenirken hata:', error);
    } finally {
      setLoading(false);
    }
  };

  const dismissNotification = (notificationId: string) => {
    const newDismissed = [...dismissedNotifications, notificationId];
    setDismissedNotifications(newDismissed);
    localStorage.setItem('dismissedNotifications', JSON.stringify(newDismissed));
  };

  const handleNotificationClick = (notification: Notifications) => {
    if (notification.redirectionLink) {
      window.open(notification.redirectionLink, '_blank');
    }
  };

  if (loading) return null;

  // Filter out dismissed notifications
  const visibleNotifications = notifications.filter(
    notification => !dismissedNotifications.includes(notification._id)
  );

  if (visibleNotifications.length === 0) return null;

  return (
    <div className={`space-y-3 ${className}`}>
      {visibleNotifications.map((notification) => (
        <Card 
          key={notification._id}
          className="relative bg-gradient-to-r from-accentneongreen/10 to-secondary/10 border-accentneongreen/30 p-4"
        >
          <div className="flex items-start justify-between gap-4">
            <div 
              className={`flex-1 ${notification.redirectionLink ? 'cursor-pointer' : ''}`}
              onClick={() => handleNotificationClick(notification)}
            >
              <div className="flex items-center gap-2 mb-2">
                <h3 className="font-heading font-semibold text-foreground">
                  {notification.title}
                </h3>
                {notification.redirectionLink && (
                  <ExternalLink className="h-4 w-4 text-secondary" />
                )}
              </div>
              <p className="font-paragraph text-sm text-foreground/80">
                {notification.content}
              </p>
            </div>
            
            <Button
              variant="ghost"
              size="sm"
              onClick={() => dismissNotification(notification._id)}
              className="h-8 w-8 p-0 hover:bg-foreground/10"
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
        </Card>
      ))}
    </div>
  );
}