import { useState, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { BaseCrudService } from '@/integrations';
import { Notifications } from '@/entities';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { X, Bell, ExternalLink, AlertCircle } from 'lucide-react';

interface NotificationPopupProps {
  isOpen: boolean;
  onClose: () => void;
}

export function NotificationPopup({ isOpen, onClose }: NotificationPopupProps) {
  const [notifications, setNotifications] = useState<Notifications[]>([]);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    if (isOpen) {
      fetchNotifications();
    }
  }, [isOpen]);

  const fetchNotifications = async () => {
    try {
      setIsLoading(true);
      const response = await BaseCrudService.getAll<Notifications>('notifications');
      // Filter active notifications and sort by display order
      const activeNotifications = response.items
        .filter(notification => notification.isActive)
        .sort((a, b) => (a.displayOrder || 0) - (b.displayOrder || 0));
      setNotifications(activeNotifications);
    } catch (error) {
      console.error('Error fetching notifications:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleNotificationClick = (notification: Notifications) => {
    if (notification.redirectionLink) {
      window.open(notification.redirectionLink, '_blank');
    }
  };

  return (
    <AnimatePresence>
      {isOpen && (
        <>
          {/* Backdrop */}
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            className="fixed inset-0 bg-black/50 backdrop-blur-sm z-50"
            onClick={onClose}
          />
          
          {/* Popup */}
          <motion.div
            initial={{ opacity: 0, scale: 0.9, y: 20 }}
            animate={{ opacity: 1, scale: 1, y: 0 }}
            exit={{ opacity: 0, scale: 0.9, y: 20 }}
            className="fixed top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 z-50 w-full max-w-2xl max-h-[80vh] overflow-hidden"
          >
            <Card className="bg-gradient-to-br from-primary via-primary/95 to-secondary/20 border border-accentneongreen/30 backdrop-blur-xl shadow-2xl">
              <CardHeader className="pb-4 border-b border-accentneongreen/20">
                <div className="flex items-center justify-between">
                  <CardTitle className="text-2xl font-bold text-primary-foreground flex items-center">
                    <div className="p-2 bg-accentneongreen/20 rounded-lg mr-3">
                      <Bell className="w-6 h-6 text-accentneongreen" />
                    </div>
                    Bildirimler
                  </CardTitle>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={onClose}
                    className="text-primary-foreground/70 hover:text-primary-foreground hover:bg-primary-foreground/10"
                  >
                    <X className="w-5 h-5" />
                  </Button>
                </div>
              </CardHeader>
              
              <CardContent className="p-0">
                <div className="max-h-[60vh] overflow-y-auto">
                  {isLoading ? (
                    <div className="p-8 text-center">
                      <div className="w-8 h-8 border-2 border-accentneongreen border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
                      <p className="text-primary-foreground/70">Bildirimler yükleniyor...</p>
                    </div>
                  ) : notifications.length === 0 ? (
                    <div className="p-8 text-center">
                      <div className="w-16 h-16 bg-accentneongreen/20 rounded-full flex items-center justify-center mx-auto mb-4">
                        <Bell className="w-8 h-8 text-accentneongreen" />
                      </div>
                      <h3 className="font-heading text-xl font-bold text-primary-foreground mb-2">
                        Henüz Bildirim Yok
                      </h3>
                      <p className="text-primary-foreground/70">
                        Şu anda görüntülenecek bildirim bulunmuyor.
                      </p>
                    </div>
                  ) : (
                    <div className="space-y-1">
                      {notifications.map((notification, index) => (
                        <motion.div
                          key={notification._id}
                          initial={{ opacity: 0, x: -20 }}
                          animate={{ opacity: 1, x: 0 }}
                          transition={{ delay: index * 0.1 }}
                          className={`p-4 border-b border-primary-foreground/10 last:border-b-0 hover:bg-primary-foreground/5 transition-colors ${
                            notification.redirectionLink ? 'cursor-pointer' : ''
                          }`}
                          onClick={() => handleNotificationClick(notification)}
                        >
                          <div className="flex items-start space-x-3">
                            <div className="flex-shrink-0 mt-1">
                              <div className="w-8 h-8 bg-accentneongreen/20 rounded-lg flex items-center justify-center">
                                <AlertCircle className="w-4 h-4 text-accentneongreen" />
                              </div>
                            </div>
                            <div className="flex-1 min-w-0">
                              <div className="flex items-start justify-between">
                                <h4 className="font-heading text-lg font-bold text-primary-foreground mb-1 line-clamp-2">
                                  {notification.title}
                                </h4>
                                {notification.redirectionLink && (
                                  <ExternalLink className="w-4 h-4 text-accentneongreen flex-shrink-0 ml-2 mt-1" />
                                )}
                              </div>
                              {notification.content && (
                                <p className="font-paragraph text-primary-foreground/80 text-sm leading-relaxed line-clamp-3">
                                  {notification.content}
                                </p>
                              )}
                              {notification.displayOrder !== undefined && (
                                <div className="mt-2">
                                  <Badge className="bg-secondary/20 text-secondary border-secondary/30 text-xs">
                                    Öncelik: {notification.displayOrder}
                                  </Badge>
                                </div>
                              )}
                            </div>
                          </div>
                        </motion.div>
                      ))}
                    </div>
                  )}
                </div>
              </CardContent>
            </Card>
          </motion.div>
        </>
      )}
    </AnimatePresence>
  );
}