import { useState, useEffect } from 'react';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Server, CheckCircle, AlertCircle, Clock, Download, Play } from 'lucide-react';
import { ServerVideoStorage } from '@/lib/server-video-storage';

interface ServerVideoStatusProps {
  videoUrl: string;
  className?: string;
  showActions?: boolean;
}

export function ServerVideoStatus({ 
  videoUrl, 
  className = '',
  showActions = false 
}: ServerVideoStatusProps) {
  const [videoInfo, setVideoInfo] = useState<any>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [streamUrl, setStreamUrl] = useState<string | null>(null);

  useEffect(() => {
    const loadVideoInfo = async () => {
      setIsLoading(true);
      
      try {
        if (ServerVideoStorage.isServerVideoUrl(videoUrl)) {
          const videoId = ServerVideoStorage.extractVideoIdFromUrl(videoUrl);
          const info = await ServerVideoStorage.getVideoInfo(videoId);
          const stream = await ServerVideoStorage.getVideoStreamUrl(videoId);
          
          setVideoInfo(info);
          setStreamUrl(stream);
        }
      } catch (error) {
        console.error('Video bilgisi yüklenemedi:', error);
      } finally {
        setIsLoading(false);
      }
    };

    loadVideoInfo();
  }, [videoUrl]);

  if (isLoading) {
    return (
      <div className={`flex items-center gap-2 ${className}`}>
        <div className="w-4 h-4 border-2 border-accentneongreen border-t-transparent rounded-full animate-spin" />
        <span className="text-sm text-secondary-foreground/60">Video durumu kontrol ediliyor...</span>
      </div>
    );
  }

  if (!videoInfo) {
    return (
      <div className={`flex items-center gap-2 ${className}`}>
        <AlertCircle className="w-4 h-4 text-red-400" />
        <Badge variant="destructive" className="text-xs">
          Video Bulunamadı
        </Badge>
      </div>
    );
  }

  const formatFileSize = (bytes: number) => {
    const mb = bytes / (1024 * 1024);
    return `${mb.toFixed(1)} MB`;
  };

  const formatUploadDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('tr-TR', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  return (
    <div className={`space-y-2 ${className}`}>
      {/* Status Badge */}
      <div className="flex items-center gap-2">
        <Server className="w-4 h-4 text-accentneongreen" />
        <Badge variant="secondary" className="bg-accentneongreen/10 text-accentneongreen border-accentneongreen/30">
          <CheckCircle className="w-3 h-3 mr-1" />
          Sunucuda Mevcut
        </Badge>
      </div>

      {/* Video Info */}
      <div className="text-xs text-secondary-foreground/60 space-y-1">
        <div className="flex items-center gap-4">
          <span>📁 {videoInfo.originalName}</span>
          <span>📊 {formatFileSize(videoInfo.size)}</span>
        </div>
        <div className="flex items-center gap-2">
          <Clock className="w-3 h-3" />
          <span>Yüklenme: {formatUploadDate(videoInfo.uploadDate)}</span>
        </div>
      </div>

      {/* Actions */}
      {showActions && streamUrl && (
        <div className="flex gap-2 pt-2">
          <Button
            size="sm"
            variant="outline"
            onClick={() => window.open(streamUrl, '_blank')}
            className="text-xs border-accentneongreen text-accentneongreen hover:bg-accentneongreen hover:text-primary"
          >
            <Play className="w-3 h-3 mr-1" />
            Oynat
          </Button>
          <Button
            size="sm"
            variant="outline"
            onClick={() => {
              const a = document.createElement('a');
              a.href = streamUrl;
              a.download = videoInfo.originalName;
              a.click();
            }}
            className="text-xs border-secondary-foreground/30 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
          >
            <Download className="w-3 h-3 mr-1" />
            İndir
          </Button>
        </div>
      )}

      {/* Technical Details (for debugging) */}
      {process.env.NODE_ENV === 'development' && (
        <details className="text-xs text-secondary-foreground/40">
          <summary className="cursor-pointer">Teknik Detaylar</summary>
          <div className="mt-1 pl-2 border-l border-secondary-foreground/20">
            <div>Video ID: {videoInfo.id}</div>
            <div>URL: {videoInfo.url}</div>
            <div>Tip: {videoInfo.type}</div>
            <div>Stream: {streamUrl ? 'Mevcut' : 'Yok'}</div>
          </div>
        </details>
      )}
    </div>
  );
}