import { useState } from 'react';
import { MangaVolumes } from '@/entities';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Image } from '@/components/ui/image';
import { Badge } from '@/components/ui/badge';
import { ExternalLink, X, Book } from 'lucide-react';

interface VolumePreviewProps {
  volume: MangaVolumes;
  isOpen: boolean;
  onClose: () => void;
}

export function VolumePreview({ volume, isOpen, onClose }: VolumePreviewProps) {
  const [imageError, setImageError] = useState(false);

  const handleReadClick = () => {
    if (volume.readingLink) {
      window.open(volume.readingLink, '_blank', 'noopener,noreferrer');
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto bg-secondary border-0">
        <DialogHeader className="flex flex-row items-center justify-between space-y-0 pb-4">
          <DialogTitle className="font-heading text-2xl font-bold text-secondary-foreground">
            {volume.volumeTitle || `Cilt ${volume.volumeNumber}`}
          </DialogTitle>
          <Button
            variant="ghost"
            size="sm"
            onClick={onClose}
            className="text-secondary-foreground/60 hover:text-secondary-foreground"
          >
            <X className="w-5 h-5" />
          </Button>
        </DialogHeader>

        <div className="space-y-6">
          {/* Volume Info */}
          <div className="flex items-center gap-4">
            {volume.volumeNumber && (
              <Badge className="bg-accentneongreen text-primary">
                Cilt {volume.volumeNumber}
              </Badge>
            )}
            <h3 className="font-paragraph text-lg text-secondary-foreground">
              {volume.volumeTitle}
            </h3>
          </div>

          {/* Preview Content */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Cover Image */}
            <div className="space-y-4">
              <h4 className="font-heading text-lg font-semibold text-secondary-foreground">
                Kapak Görseli
              </h4>
              <div className="aspect-[3/4] relative overflow-hidden rounded-xl bg-primary/10">
                {volume.coverImage && !imageError ? (
                  <Image
                    src={volume.coverImage}
                    alt={volume.volumeTitle || `Cilt ${volume.volumeNumber} kapağı`}
                    className="w-full h-full object-cover"
                    width={300}
                    onError={() => setImageError(true)}
                  />
                ) : (
                  <div className="w-full h-full flex items-center justify-center">
                    <Book className="w-16 h-16 text-secondary-foreground/30" />
                  </div>
                )}
              </div>
            </div>

            {/* Preview Frame */}
            <div className="space-y-4">
              <h4 className="font-heading text-lg font-semibold text-secondary-foreground">
                Önizleme
              </h4>
              <div className="aspect-[4/3] relative overflow-hidden rounded-xl bg-primary/10">
                {volume.previewUrl ? (
                  <iframe
                    src={volume.previewUrl}
                    className="w-full h-full border-0"
                    title={`${volume.volumeTitle} önizlemesi`}
                    sandbox="allow-scripts allow-same-origin"
                  />
                ) : (
                  <div className="w-full h-full flex items-center justify-center">
                    <div className="text-center space-y-2">
                      <Book className="w-12 h-12 text-secondary-foreground/30 mx-auto" />
                      <p className="font-paragraph text-secondary-foreground/60">
                        Bu cilt için önizleme mevcut değil
                      </p>
                    </div>
                  </div>
                )}
              </div>
            </div>
          </div>

          {/* Action Buttons */}
          <div className="flex gap-4 pt-4 border-t border-secondary-foreground/10">
            <Button
              onClick={handleReadClick}
              disabled={!volume.readingLink}
              className="flex-1 bg-accentneongreen text-primary hover:bg-accentneongreen/90"
            >
              <ExternalLink className="w-4 h-4 mr-2" />
              Okumaya Başla
            </Button>
            <Button
              variant="outline"
              onClick={onClose}
              className="border-secondary-foreground/20 text-secondary-foreground hover:bg-secondary-foreground hover:text-secondary"
            >
              Kapat
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}