import { useState, useEffect } from 'react';
import { BaseCrudService } from '@/integrations';
import { YorumBeenileri } from '@/entities';
import { useMember } from '@/integrations';
import type { Member } from '@/types/member';

export const useCommentLikes = () => {
  const { member } = useMember();
  const [commentLikes, setCommentLikes] = useState<YorumBeenileri[]>([]);
  const [loading, setLoading] = useState(false);

  // Yorum beğen/beğenme
  const toggleCommentLike = async (commentId: string, likeType: 'like' | 'dislike') => {
    if (!member?.loginEmail) return;
    
    try {
      setLoading(true);
      
      // Mevcut beğeniyi kontrol et
      const { items } = await BaseCrudService.getAll<YorumBeenileri>('yorumbeenileri');
      const existingLike = items.find(
        like => like.userId === member.loginEmail && like.commentId === commentId
      );

      if (existingLike) {
        if (existingLike.likeType === likeType) {
          // Aynı türde beğeni varsa kaldır
          await BaseCrudService.delete('yorumbeenileri', existingLike._id);
        } else {
          // Farklı türde beğeni varsa güncelle
          await BaseCrudService.update('yorumbeenileri', {
            _id: existingLike._id,
            likeType,
            actionDate: new Date(),
            status: 'aktif'
          });
        }
      } else {
        // Yeni beğeni ekle
        await BaseCrudService.create('yorumbeenileri', {
          _id: crypto.randomUUID(),
          userId: member.loginEmail,
          commentId,
          likeType,
          actionDate: new Date(),
          status: 'aktif'
        });
      }
      
      await loadCommentLikes();
    } catch (error) {
      console.error('Yorum beğeni işlemi başarısız:', error);
    } finally {
      setLoading(false);
    }
  };

  // Belirli bir yorumun beğeni sayılarını al
  const getCommentLikeStats = (commentId: string) => {
    const likes = commentLikes.filter(
      like => like.commentId === commentId && like.status === 'aktif'
    );
    
    const likeCount = likes.filter(like => like.likeType === 'like').length;
    const dislikeCount = likes.filter(like => like.likeType === 'dislike').length;
    
    return { likeCount, dislikeCount };
  };

  // Kullanıcının bir yoruma verdiği beğeniyi al
  const getUserLikeForComment = (commentId: string): 'like' | 'dislike' | null => {
    if (!member?.loginEmail) return null;
    
    const userLike = commentLikes.find(
      like => 
        like.userId === member.loginEmail && 
        like.commentId === commentId && 
        like.status === 'aktif'
    );
    
    return userLike ? (userLike.likeType as 'like' | 'dislike') : null;
  };

  // Tüm beğenileri yükle
  const loadCommentLikes = async () => {
    try {
      const { items } = await BaseCrudService.getAll<YorumBeenileri>('yorumbeenileri');
      setCommentLikes(items);
    } catch (error) {
      console.error('Yorum beğenileri yüklenemedi:', error);
    }
  };

  useEffect(() => {
    loadCommentLikes();
  }, []);

  return {
    commentLikes,
    loading,
    toggleCommentLike,
    getCommentLikeStats,
    getUserLikeForComment,
    loadCommentLikes
  };
};