import { useState, useEffect } from 'react';
import { BaseCrudService } from '@/integrations';
import { Favoriler } from '@/entities';

export function useFavorites(userId?: string) {
  const [favorites, setFavorites] = useState<Favoriler[]>([]);
  const [isLoading, setIsLoading] = useState(false);

  useEffect(() => {
    if (userId) {
      fetchFavorites();
    }
  }, [userId]);

  const fetchFavorites = async () => {
    if (!userId) return;
    
    setIsLoading(true);
    try {
      const response = await BaseCrudService.getAll<Favoriler>('favoriler');
      const userFavorites = response.items.filter(fav => fav.userId === userId);
      setFavorites(userFavorites);
    } catch (error) {
      console.error('Favoriler yüklenirken hata:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const addToFavorites = async (contentId: string, contentType: 'anime' | 'manga') => {
    if (!userId) return false;

    try {
      // Check if already in favorites
      const existingFavorite = favorites.find(
        fav => fav.contentId === contentId && fav.contentType === contentType
      );

      if (existingFavorite) {
        return false; // Already in favorites
      }

      const favoriteData: Favoriler = {
        _id: crypto.randomUUID(),
        userId,
        contentId,
        contentType,
        addedDate: new Date().toISOString(),
        _createdDate: new Date(),
        _updatedDate: new Date()
      };

      await BaseCrudService.create('favoriler', favoriteData);
      await fetchFavorites(); // Refresh favorites
      return true;
    } catch (error) {
      console.error('Favorilere eklenirken hata:', error);
      return false;
    }
  };

  const removeFromFavorites = async (contentId: string, contentType: 'anime' | 'manga') => {
    if (!userId) return false;

    try {
      const favoriteToRemove = favorites.find(
        fav => fav.contentId === contentId && fav.contentType === contentType
      );

      if (!favoriteToRemove) {
        return false; // Not in favorites
      }

      await BaseCrudService.delete('favoriler', favoriteToRemove._id);
      await fetchFavorites(); // Refresh favorites
      return true;
    } catch (error) {
      console.error('Favorilerden çıkarılırken hata:', error);
      return false;
    }
  };

  const isFavorite = (contentId: string, contentType: 'anime' | 'manga') => {
    return favorites.some(
      fav => fav.contentId === contentId && fav.contentType === contentType
    );
  };

  return {
    favorites,
    isLoading,
    addToFavorites,
    removeFromFavorites,
    isFavorite,
    refreshFavorites: fetchFavorites
  };
}