import { useState, useEffect } from 'react';
import { BaseCrudService } from '@/integrations';
import { Arkadalkstekleri, Arkadalklar, ZelMesajlar } from '@/entities';
import { useMember } from '@/integrations';
import type { Member } from '@/types/member';

export const useFriendship = () => {
  const { member }: { member: Member | null } = useMember();
  const [friendRequests, setFriendRequests] = useState<Arkadalkstekleri[]>([]);
  const [friends, setFriends] = useState<Arkadalklar[]>([]);
  const [messages, setMessages] = useState<ZelMesajlar[]>([]);
  const [loading, setLoading] = useState(false);

  // Arkadaşlık isteği gönder
  const sendFriendRequest = async (receiverUserId: string, message?: string) => {
    if (!member?._id) return;
    
    try {
      setLoading(true);
      await BaseCrudService.create('arkadalkistekleri', {
        _id: crypto.randomUUID(),
        senderUserId: member._id,
        receiverUserId,
        status: 'beklemede',
        requestDate: new Date(),
        message: message || ''
      });
      await loadFriendRequests();
    } catch (error) {
      console.error('Arkadaşlık isteği gönderilemedi:', error);
    } finally {
      setLoading(false);
    }
  };

  // Arkadaşlık isteğini kabul et
  const acceptFriendRequest = async (requestId: string, senderUserId: string) => {
    if (!member?._id) return;
    
    try {
      setLoading(true);
      // İsteği kabul edildi olarak güncelle
      await BaseCrudService.update('arkadalkistekleri', {
        _id: requestId,
        status: 'kabul edildi'
      });
      
      // Arkadaşlık kaydı oluştur
      await BaseCrudService.create('arkadalklar', {
        _id: crypto.randomUUID(),
        userOneId: senderUserId,
        userTwoId: member._id,
        friendshipDate: new Date(),
        status: 'aktif'
      });
      
      await loadFriendRequests();
      await loadFriends();
    } catch (error) {
      console.error('Arkadaşlık isteği kabul edilemedi:', error);
    } finally {
      setLoading(false);
    }
  };

  // Arkadaşlık isteğini reddet
  const rejectFriendRequest = async (requestId: string) => {
    try {
      setLoading(true);
      await BaseCrudService.update('arkadalkistekleri', {
        _id: requestId,
        status: 'reddedildi'
      });
      await loadFriendRequests();
    } catch (error) {
      console.error('Arkadaşlık isteği reddedilemedi:', error);
    } finally {
      setLoading(false);
    }
  };

  // Özel mesaj gönder
  const sendMessage = async (recipientId: string, messageContent: string) => {
    if (!member?._id) return;
    
    try {
      setLoading(true);
      await BaseCrudService.create('zelmesajlar', {
        _id: crypto.randomUUID(),
        senderId: member._id,
        recipientId,
        messageContent,
        sentAt: new Date(),
        isRead: false
      });
      await loadMessages(recipientId);
    } catch (error) {
      console.error('Mesaj gönderilemedi:', error);
    } finally {
      setLoading(false);
    }
  };

  // Arkadaşlık isteklerini yükle
  const loadFriendRequests = async () => {
    if (!member?._id) return;
    
    try {
      const { items } = await BaseCrudService.getAll<Arkadalkstekleri>('arkadalkistekleri');
      const userRequests = items.filter(
        req => req.receiverUserId === member._id && req.status === 'beklemede'
      );
      setFriendRequests(userRequests);
    } catch (error) {
      console.error('Arkadaşlık istekleri yüklenemedi:', error);
    }
  };

  // Arkadaşları yükle
  const loadFriends = async () => {
    if (!member?._id) return;
    
    try {
      const { items } = await BaseCrudService.getAll<Arkadalklar>('arkadalklar');
      const userFriends = items.filter(
        friendship => 
          (friendship.userOneId === member._id || friendship.userTwoId === member._id) &&
          friendship.status === 'aktif'
      );
      setFriends(userFriends);
    } catch (error) {
      console.error('Arkadaşlar yüklenemedi:', error);
    }
  };

  // Mesajları yükle
  const loadMessages = async (otherUserId: string) => {
    if (!member?._id) return;
    
    try {
      const { items } = await BaseCrudService.getAll<ZelMesajlar>('zelmesajlar');
      const conversation = items.filter(
        msg => 
          (msg.senderId === member._id && msg.recipientId === otherUserId) ||
          (msg.senderId === otherUserId && msg.recipientId === member._id)
      ).sort((a, b) => new Date(a.sentAt || 0).getTime() - new Date(b.sentAt || 0).getTime());
      setMessages(conversation);
    } catch (error) {
      console.error('Mesajlar yüklenemedi:', error);
    }
  };

  // Mesajı okundu olarak işaretle
  const markAsRead = async (messageId: string) => {
    try {
      await BaseCrudService.update('zelmesajlar', {
        _id: messageId,
        isRead: true
      });
    } catch (error) {
      console.error('Mesaj okundu olarak işaretlenemedi:', error);
    }
  };

  // İki kullanıcının arkadaş olup olmadığını kontrol et
  const areFriends = (userId: string): boolean => {
    if (!member?._id) return false;
    return friends.some(
      friendship => 
        (friendship.userOneId === member._id && friendship.userTwoId === userId) ||
        (friendship.userOneId === userId && friendship.userTwoId === member._id)
    );
  };

  useEffect(() => {
    if (member?._id) {
      loadFriendRequests();
      loadFriends();
    }
  }, [member?._id]);

  return {
    friendRequests,
    friends,
    messages,
    loading,
    sendFriendRequest,
    acceptFriendRequest,
    rejectFriendRequest,
    sendMessage,
    loadMessages,
    markAsRead,
    areFriends,
    loadFriendRequests,
    loadFriends
  };
};