// Server-based video storage utility for Wix platform
export class ServerVideoStorage {
  private static baseUrl = 'https://static.wixstatic.com/media';

  // Upload video to Wix Media Manager (simulated)
  static async uploadVideo(file: File, metadata: {
    title: string;
    description?: string;
    category?: string;
    animeId?: string;
    episodeNumber?: number;
  }): Promise<{ videoId: string; videoUrl: string; thumbnailUrl: string }> {
    
    // Dosya doğrulama
    if (!file) {
      throw new Error('Dosya seçilmedi');
    }

    // Desteklenen formatları kontrol et
    const supportedFormats = [
      'video/mp4',
      'video/webm', 
      'video/ogg',
      'video/avi',
      'video/mov',
      'video/quicktime'
    ];
    
    const fileName = file.name.toLowerCase();
    const supportedExtensions = ['.mp4', '.webm', '.ogg', '.avi', '.mov'];
    const hasValidExtension = supportedExtensions.some(ext => fileName.endsWith(ext));
    
    if (!supportedFormats.includes(file.type) && !hasValidExtension) {
      throw new Error('Desteklenmeyen video formatı. Sadece MP4, WebM, OGG, AVI ve MOV formatları kabul edilir.');
    }

    // Dosya boyutu kontrolü (tüm videolar için 2GB)
    const maxSize = 2 * 1024 * 1024 * 1024; // 2GB
    if (file.size > maxSize) {
      throw new Error('Dosya boyutu 2GB limitini aşıyor');
    }

    if (file.size < 1024) {
      throw new Error('Dosya çok küçük, geçerli bir video dosyası değil');
    }

    // Simulate upload progress
    return new Promise((resolve, reject) => {
      try {
        // Generate unique ID for the video
        const videoId = crypto.randomUUID();
        
        // Create Wix-style media URL (simulated server storage)
        const mediaHash = this.generateMediaHash();
        const videoUrl = `${this.baseUrl}/${mediaHash}~mv2.mp4`;
        const thumbnailUrl = `${this.baseUrl}/${mediaHash}_thumb~mv2.jpg`;
        
        // Store video metadata in localStorage (simulating server database)
        const videoData = {
          id: videoId,
          url: videoUrl,
          thumbnailUrl: thumbnailUrl,
          originalName: file.name,
          size: file.size,
          type: file.type,
          uploadDate: new Date().toISOString(),
          metadata,
          // Store actual file data as base64 for demo purposes
          fileData: null // Will be set below
        };

        // Convert file to base64 for storage
        const reader = new FileReader();
        reader.onload = (e) => {
          videoData.fileData = e.target?.result as string;
          
          // Store in localStorage (simulating server storage)
          const existingVideos = JSON.parse(localStorage.getItem('wixServerVideos') || '[]');
          existingVideos.push(videoData);
          localStorage.setItem('wixServerVideos', JSON.stringify(existingVideos));
          
          resolve({ 
            videoId, 
            videoUrl,
            thumbnailUrl
          });
        };
        
        reader.onerror = () => {
          reject(new Error('Video dosyası işlenirken hata oluştu'));
        };
        
        reader.readAsDataURL(file);
        
      } catch (error) {
        reject(new Error('Video yükleme başarısız oldu'));
      }
    });
  }

  // Generate Wix-style media hash
  private static generateMediaHash(): string {
    const chars = 'abcdefghijklmnopqrstuvwxyz0123456789';
    let hash = '';
    for (let i = 0; i < 32; i++) {
      hash += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return `12d367_${hash}`;
  }

  // Get video URL from server
  static async getVideoUrl(videoId: string): Promise<string | null> {
    try {
      const existingVideos = JSON.parse(localStorage.getItem('wixServerVideos') || '[]');
      const video = existingVideos.find((v: any) => v.id === videoId);
      
      if (video && video.fileData) {
        // Return blob URL created from stored data
        const blob = this.dataURLToBlob(video.fileData);
        return URL.createObjectURL(blob);
      }
      
      return null;
    } catch (error) {
      console.error('Video URL alınamadı:', error);
      return null;
    }
  }

  // Convert data URL to blob
  private static dataURLToBlob(dataURL: string): Blob {
    const arr = dataURL.split(',');
    const mime = arr[0].match(/:(.*?);/)![1];
    const bstr = atob(arr[1]);
    let n = bstr.length;
    const u8arr = new Uint8Array(n);
    while (n--) {
      u8arr[n] = bstr.charCodeAt(n);
    }
    return new Blob([u8arr], { type: mime });
  }

  // Delete video from server
  static async deleteVideo(videoId: string): Promise<boolean> {
    try {
      const existingVideos = JSON.parse(localStorage.getItem('wixServerVideos') || '[]');
      const filteredVideos = existingVideos.filter((v: any) => v.id !== videoId);
      localStorage.setItem('wixServerVideos', JSON.stringify(filteredVideos));
      return true;
    } catch (error) {
      console.error('Video silinemedi:', error);
      return false;
    }
  }

  // List all videos
  static async listVideos(): Promise<Array<{
    id: string;
    url: string;
    thumbnailUrl: string;
    originalName: string;
    size: number;
    type: string;
    uploadDate: string;
    metadata: any;
  }>> {
    try {
      const existingVideos = JSON.parse(localStorage.getItem('wixServerVideos') || '[]');
      return existingVideos.map((video: any) => ({
        id: video.id,
        url: video.url,
        thumbnailUrl: video.thumbnailUrl,
        originalName: video.originalName,
        size: video.size,
        type: video.type,
        uploadDate: video.uploadDate,
        metadata: video.metadata
      }));
    } catch (error) {
      console.error('Video listesi alınamadı:', error);
      return [];
    }
  }

  // Check if URL is a server video URL
  static isServerVideoUrl(url: string): boolean {
    return url.startsWith(this.baseUrl) || url.startsWith('/api/videos/');
  }

  // Extract video ID from server URL
  static extractVideoIdFromUrl(url: string): string {
    // Extract from Wix media URL or API URL
    if (url.includes('/api/videos/')) {
      return url.split('/api/videos/')[1];
    }
    // For Wix media URLs, we need to find the video by URL
    try {
      const existingVideos = JSON.parse(localStorage.getItem('wixServerVideos') || '[]');
      const video = existingVideos.find((v: any) => v.url === url);
      return video ? video.id : '';
    } catch {
      return '';
    }
  }

  // Create server video URL
  static createServerVideoUrl(videoId: string): string {
    return `/api/videos/${videoId}`;
  }

  // Get video file info
  static async getVideoInfo(videoId: string): Promise<{
    id: string;
    originalName: string;
    size: number;
    type: string;
    uploadDate: string;
    metadata: any;
    url: string;
    thumbnailUrl: string;
  } | null> {
    try {
      const existingVideos = JSON.parse(localStorage.getItem('wixServerVideos') || '[]');
      const video = existingVideos.find((v: any) => v.id === videoId);
      
      if (video) {
        return {
          id: video.id,
          originalName: video.originalName,
          size: video.size,
          type: video.type,
          uploadDate: video.uploadDate,
          metadata: video.metadata,
          url: video.url,
          thumbnailUrl: video.thumbnailUrl
        };
      }
      
      return null;
    } catch (error) {
      console.error('Video bilgisi alınamadı:', error);
      return null;
    }
  }

  // Get video stream URL for playback
  static async getVideoStreamUrl(videoId: string): Promise<string | null> {
    try {
      const existingVideos = JSON.parse(localStorage.getItem('wixServerVideos') || '[]');
      const video = existingVideos.find((v: any) => v.id === videoId);
      
      if (video && video.fileData) {
        // Create blob URL for streaming
        const blob = this.dataURLToBlob(video.fileData);
        return URL.createObjectURL(blob);
      }
      
      return null;
    } catch (error) {
      console.error('Video stream URL alınamadı:', error);
      return null;
    }
  }

  // Check video availability
  static async isVideoAvailable(videoId: string): Promise<boolean> {
    try {
      const existingVideos = JSON.parse(localStorage.getItem('wixServerVideos') || '[]');
      return existingVideos.some((v: any) => v.id === videoId);
    } catch (error) {
      return false;
    }
  }

  // Get video thumbnail
  static async getVideoThumbnail(videoId: string): Promise<string | null> {
    try {
      const existingVideos = JSON.parse(localStorage.getItem('wixServerVideos') || '[]');
      const video = existingVideos.find((v: any) => v.id === videoId);
      
      if (video) {
        return video.thumbnailUrl;
      }
      
      return null;
    } catch (error) {
      console.error('Video thumbnail alınamadı:', error);
      return null;
    }
  }
}