// Video storage utility for IndexedDB
export class VideoStorage {
  private static dbName = 'AnimeVideos';
  private static version = 1;
  private static storeName = 'videos';

  static async openDB(): Promise<IDBDatabase> {
    return new Promise((resolve, reject) => {
      const request = indexedDB.open(this.dbName, this.version);
      
      request.onupgradeneeded = (event) => {
        const db = (event.target as IDBOpenDBRequest).result;
        if (!db.objectStoreNames.contains(this.storeName)) {
          db.createObjectStore(this.storeName, { keyPath: 'id' });
        }
      };
      
      request.onsuccess = (event) => {
        resolve((event.target as IDBOpenDBRequest).result);
      };
      
      request.onerror = (event) => {
        reject((event.target as IDBOpenDBRequest).error);
      };
    });
  }

  static async storeVideo(id: string, file: File): Promise<void> {
    const db = await this.openDB();
    const arrayBuffer = await file.arrayBuffer();
    
    const transaction = db.transaction([this.storeName], 'readwrite');
    const store = transaction.objectStore(this.storeName);
    
    const videoData = {
      id,
      data: arrayBuffer,
      type: file.type,
      name: file.name,
      size: file.size,
      uploadDate: new Date().toISOString()
    };
    
    return new Promise((resolve, reject) => {
      const request = store.add(videoData);
      request.onsuccess = () => resolve();
      request.onerror = () => reject(request.error);
    });
  }

  static async getVideo(id: string): Promise<Blob | null> {
    try {
      const db = await this.openDB();
      const transaction = db.transaction([this.storeName], 'readonly');
      const store = transaction.objectStore(this.storeName);
      
      return new Promise((resolve, reject) => {
        const request = store.get(id);
        request.onsuccess = () => {
          const result = request.result;
          if (result) {
            const blob = new Blob([result.data], { type: result.type });
            resolve(blob);
          } else {
            resolve(null);
          }
        };
        request.onerror = () => reject(request.error);
      });
    } catch (error) {
      console.error('Error retrieving video from IndexedDB:', error);
      return null;
    }
  }

  static async deleteVideo(id: string): Promise<void> {
    const db = await this.openDB();
    const transaction = db.transaction([this.storeName], 'readwrite');
    const store = transaction.objectStore(this.storeName);
    
    return new Promise((resolve, reject) => {
      const request = store.delete(id);
      request.onsuccess = () => resolve();
      request.onerror = () => reject(request.error);
    });
  }

  static async listVideos(): Promise<string[]> {
    const db = await this.openDB();
    const transaction = db.transaction([this.storeName], 'readonly');
    const store = transaction.objectStore(this.storeName);
    
    return new Promise((resolve, reject) => {
      const request = store.getAllKeys();
      request.onsuccess = () => resolve(request.result as string[]);
      request.onerror = () => reject(request.error);
    });
  }

  static createVideoUrl(id: string): string {
    return `indexeddb://${id}`;
  }

  static isIndexedDBUrl(url: string): boolean {
    return url.startsWith('indexeddb://');
  }

  static extractIdFromUrl(url: string): string {
    return url.replace('indexeddb://', '');
  }
}